/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.sql.SQLException;
import java.time.Duration;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BatchStatement;
import com.datastax.oss.driver.api.core.cql.DefaultBatchType;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;

import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.delivery.conversationreference.repository.ConversationReference;
import net.bluemind.delivery.conversationreference.repository.IConversationReferenceStore;

public class CqlConvRefStore extends CqlAbstractStore implements IConversationReferenceStore {

	private static final int ONE_YEAR_TTL = (int) Duration.ofDays(365).toSeconds();
	private static final Logger logger = LoggerFactory.getLogger(CqlConvRefStore.class);

	public CqlConvRefStore(CqlSession s) {
		super(s);
	}

	@Override
	public long get(Long mailboxId, List<Long> references) throws SQLException {

		ConversationReference found = unique("""
				SELECT message_id_hash, conversation_id
				FROM t_conversation_reference
				WHERE subtree_id=? AND message_id_hash IN ?
				LIMIT 1
				""", //
				r -> new ConversationReference(r.getLong(0), r.getLong(1), mailboxId), voidPop(), //
				mailboxId, references);
		return found == null ? 0L : found.conversationId;
	}

	@Override
	public void create(List<ConversationReference> crs) throws SQLException {
		BatchStatement bs = BatchStatement.newInstance(DefaultBatchType.LOGGED);
		PreparedStatement ps = session.prepare("""
				INSERT INTO t_conversation_reference
				(subtree_id, message_id_hash, conversation_id)
				VALUES (?,?,?)
				USING TTL ?
				""");
		bs = bs.addAll(crs.stream().map(cr -> ps.bind(cr.mailboxId, cr.messageIdHash, cr.conversationId, ONE_YEAR_TTL))
				.toList());
		session.execute(bs);
		logger.info("Persisted {} reference(s)", crs.size());
	}

}
