/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.shared.mailbox.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.core.task.service.TaskUtils;
import net.bluemind.dataprotect.addressbook.RestoreUserAddressbooks;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.RestoreDefinition;
import net.bluemind.dataprotect.calendar.RestoreUserCalendars;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.directory.RestoreDirWorker;
import net.bluemind.dataprotect.mailbox.AbstractRestoreActionProvider;
import net.bluemind.dataprotect.notes.RestoreUserNotes;
import net.bluemind.dataprotect.todolist.RestoreUserTodolists;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.MailFilter;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.TagChanges;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.User;

public class SharedMailboxRestoreWorker extends RestoreDirWorker<User> {
	private static final Logger logger = LoggerFactory.getLogger(SharedMailboxRestoreWorker.class);

	public SharedMailboxRestoreWorker(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		super(backup, restorableItem);
	}

	@Override
	public void restoreEntryDependantContainers() {
		logger.info("Restoring shared mailbox mbox {}({}):{}", restorableItem.entryUid(), restorableItem.liveEntryUid(),
				restorableItem.displayName());

		RestoreDefinition restoreBox = new RestoreDefinition(AbstractRestoreActionProvider.RESTORE_REPLACE_IDENTIFIER,
				backup.id, restorableItem.item());
		executeTask(restoreBox);

		logger.info("Restoring shared mailbox addressbooks {}:{}", restorableItem.entryUid(),
				restorableItem.displayName());

		RestoreDefinition restoreBooks = new RestoreDefinition(RestoreUserAddressbooks.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreBooks);

		logger.info("Restoring shared mailbox calendars {}:{}", restorableItem.entryUid(),
				restorableItem.displayName());

		RestoreDefinition restoreCalendars = new RestoreDefinition(RestoreUserCalendars.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreCalendars);

		logger.info("Restoring shared mailbox todolists {}:{}", restorableItem.entryUid(),
				restorableItem.displayName());

		RestoreDefinition restoreTodolists = new RestoreDefinition(RestoreUserTodolists.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreTodolists);

		logger.info("Restoring shared mailbox notes {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		RestoreDefinition restoreNotes = new RestoreDefinition(RestoreUserNotes.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreNotes);
	}

	@Override
	public ItemValue<User> createOrUpdateLiveEntry(ItemValue<User> backupSharedMailbox) throws InterruptedException {
		IUser sharedMailboxServiceLive = live.instance(IUser.class, restorableItem.domain());

		ItemValue<User> liveSharedMailbox = sharedMailboxServiceLive.getComplete(restorableItem.entryUid());
		if (liveSharedMailbox != null) {
			sharedMailboxServiceLive.update(liveSharedMailbox.uid, backupSharedMailbox.value);
			restorableItem.setLiveEntryUid(liveSharedMailbox.uid);
			return liveSharedMailbox;
		}

		liveSharedMailbox = sharedMailboxServiceLive.byLogin(backupSharedMailbox.value.login);
		if (liveSharedMailbox != null) {
			TaskRef deleteTask = sharedMailboxServiceLive.delete(liveSharedMailbox.uid);
			TaskStatus taskDeleteStatus = TaskUtils.waitForInterruptible(live, deleteTask);
			if (taskDeleteStatus.state.succeed) {
				restorableItem.monitorLog("Delete shared mailbox '{}' succeed {}", backupSharedMailbox.value.login,
						taskDeleteStatus.result);
			}
		}

		sharedMailboxServiceLive.create(restorableItem.entryUid(), backupSharedMailbox.value);
		liveSharedMailbox = sharedMailboxServiceLive.getComplete(restorableItem.entryUid());
		restorableItem.setLiveEntryUid(liveSharedMailbox.uid);
		return liveSharedMailbox;
	}

	public void restoreSharedMailboxFilters(MailFilter mailFilter) {
		live.instance(IMailboxes.class, restorableItem.domain()).setMailboxFilter(restorableItem.liveEntryUid(),
				mailFilter);
	}

	public void restoreSharedMailboxTags(List<TagChanges.ItemAdd> sharedMailboxTags) {
		live.instance(ITags.class, ITagUids.TYPE + "_" + restorableItem.liveEntryUid())
				.updates(TagChanges.create(sharedMailboxTags, Collections.emptyList(), Collections.emptyList()));
	}

	public void restoreSharedMailboxSettings(Map<String, String> settings) {
		live.instance(IUserSettings.class, restorableItem.domain()).set(restorableItem.liveEntryUid(), settings);
	}

	public void restoreGroupMembership(List<ItemValue<Group>> memberOf) {
		IGroup groupService = live.instance(IGroup.class, restorableItem.domain());
		for (ItemValue<Group> group : memberOf) {
			// add to group, if group still exists
			ItemValue<Group> groupItem = groupService.getComplete(group.uid);
			if (groupItem != null
					&& !groupService.getMembers(group.uid).contains(Member.user(restorableItem.liveEntryUid()))) {
				groupService.add(group.uid, Arrays.asList(Member.user(restorableItem.liveEntryUid())));
			}
		}
	}

}
