/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.command.ping;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.EventBus;
import net.bluemind.eas.dto.EasBusEndpoints;
import net.bluemind.eas.dto.ping.PingResponse;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.lib.vertx.VertxPlatform;

public class PingResponseHandler implements Handler<PingResponse> {
	private static Map<String, Optional<Handler<PingResponse>>> responseHandlersByAddress = new ConcurrentHashMap<>();

	protected final Handler<PingResponse> responseHandler;
	private String pushKillerAddress;

	final EventBus eventBus;
	final Vertx vertx;

	PingResponseHandler(Handler<PingResponse> responseHandler, BackendSession bs) {
		this.responseHandler = responseHandler;
		this.eventBus = VertxPlatform.eventBus();
		this.vertx = VertxPlatform.getVertx();
		this.pushKillerAddress = getPushKillerAddress(bs);
	}

	private static String getPushKillerAddress(BackendSession bs) {
		return EasBusEndpoints.PUSH_KILLER + "." + bs.getUniqueIdentifier();
	}

	void registerPushKiller(long timer, AtomicBoolean responseSent) {
		Optional<Handler<PingResponse>> previousHandler = getPushKillerHandler();
		responseHandlersByAddress.put(pushKillerAddress, Optional.of(responseHandler));
		if (previousHandler == null) {
			eventBus.consumer(pushKillerAddress).handler(msg -> {
				if (!responseSent.getAndSet(true)) {
					getPushKillerHandler().ifPresent(pushkillerHandler -> {
						vertx.cancelTimer(timer);
						pushkillerHandler.handle(PingResponse.createNoChangesPingResponse());
						unregisterPushKiller();
					});
				}
				msg.reply("ok");
			});
		}
	}

	private void unregisterPushKiller() {
		responseHandlersByAddress.put(pushKillerAddress, Optional.empty());
	}

	private Optional<Handler<PingResponse>> getPushKillerHandler() {
		return responseHandlersByAddress.get(pushKillerAddress);
	}

	@Override
	public void handle(PingResponse event) {
		unregisterPushKiller();
		responseHandler.handle(event);
	}
}
