/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2020
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.addressbook.service;

import java.sql.SQLException;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.domainbook.DomainAddressBook;
import net.bluemind.addressbook.persistence.VCardStore;
import net.bluemind.addressbook.service.internal.AddressBookService;
import net.bluemind.addressbook.service.internal.VCardContainerStoreService;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.service.internal.ItemValueAuditLogService;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.lib.elasticsearch.ESearchActivator;
import net.bluemind.repository.provider.RepositoryProvider;

public class InCoreAddressBookFactory
		implements ServerSideServiceProvider.IServerSideServiceFactory<IInCoreAddressBook> {

	private static final Logger logger = LoggerFactory.getLogger(InCoreAddressBookFactory.class);

	@Override
	public Class<IInCoreAddressBook> factoryClass() {
		return IInCoreAddressBook.class;
	}

	private IInCoreAddressBook getService(BmContext context, String containerId) throws ServerFault {

		IContainerRouteStore router = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation loc = router.routeOf(ContainerUid.of(containerId));
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, loc);

		Container container = null;
		try {
			container = containerStore.get(containerId);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		if (container == null) {
			throw new ServerFault("container " + containerId + " not found", ErrorCode.NOT_FOUND);
		}

		if (!container.type.equals(IAddressBookUids.TYPE)) {
			throw new ServerFault("Incompatible addressbook container: " + container.type + ", uid: " + container.uid);
		}

		if (isDefaultDomainAb(container)) {
			if (!loc.equals(DataLocation.directory())) {
				logger.warn("wrong datasource (should be bj) container.uid {}", container.uid);
			}
		} else {
			if (loc.equals(DataLocation.directory())) {
				logger.warn("wrong datasource (should be bj-data) container.uid {}", container.uid);
			}
		}

		ElasticsearchClient esClient = ESearchActivator.getClient();
		if (esClient == null) {
			throw new ServerFault("elasticsearch was not found for contact indexing");
		}

		DataSource ds = DataSourceRouter.get(context, container.uid);
		VCardStore vcardStore = new VCardStore(ds, container);
		BaseContainerDescriptor descriptor = BaseContainerDescriptor.create(container.uid, container.name,
				container.owner, container.type, container.domainUid, container.defaultContainer);
		descriptor.internalId = container.id;

		ItemValueAuditLogService<VCard> logService = new ItemValueAuditLogService<>(context, descriptor);

		VCardContainerStoreService storeService = new VCardContainerStoreService(context, context.getSecurityContext(),
				container, vcardStore, logService);
		return new AddressBookService(ds, esClient, container, context, vcardStore, storeService);
	}

	@Override
	public IInCoreAddressBook instance(BmContext context, String... params) throws ServerFault {
		if (params == null || params.length < 1) {
			throw new ServerFault("wrong number of instance parameters");
		}
		return getService(context, params[0]);
	}

	private static boolean isDefaultDomainAb(Container container) {
		return container.uid.equals(DomainAddressBook.getIdentifier(container.domainUid));
	}
}
