/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2016
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.addressbook.impl.user;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.addressbook.impl.AbstractRestoreBook;
import net.bluemind.dataprotect.addressbook.impl.AddressBookBackupRepository;
import net.bluemind.dataprotect.addressbook.impl.CommonRestoreBooks;
import net.bluemind.dataprotect.addressbook.impl.pg.RestoreUserBooksTaskPg;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.user.api.User;

public class RestoreUserBooksTask extends AbstractRestoreBook implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreUserBooksTask.class);

	protected final RestoreUserBooksWorker userBooksWorker;

	public RestoreUserBooksTask(DataProtectGeneration backup, Restorable item) {
		super(backup, item);
		this.userBooksWorker = new RestoreUserBooksWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.ADDRESSBOOK.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting restore for uid {}", restorableItem.entryUid());
		}

		AddressBookBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreUserBooksTaskPg(backup, restorableItem).run();
		}
	}

	private void restore(AddressBookBackupRepository repository) throws Exception {
		try {
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
			// For users, iterate over all available address books
			List<String> addressBookUids = repository.getRestorableUserAddressBookUids(restorableItem.domain(),
					restorableItem.entryUid());
			for (String addressBookUid : addressBookUids) {
				try (RestorableContainerItemBackupItem<User> restorable = repository.getRestorableUserAddressBook(
						restorableItem.domain(), restorableItem.entryUid(), addressBookUid)) {
					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
					AddressBookDescriptor backupAddressBook = AddressBookDescriptor.create(cd.name, cd.owner,
							cd.domainUid, cd.settings);

					List<ItemValue<VCard>> vcards = importVcf(restorable, cd.owner, cd.domainUid,
							restorableItem.item().kind);
					restorableItem.monitorLog("Restore addressbook {}@{} ({})", backupAddressBook.name,
							backupAddressBook.domainUid, addressBookUid);

					userBooksWorker.createOrUpdateLiveEntry(ItemValue.create(addressBookUid, backupAddressBook));

					new CommonRestoreBooks(restorableItem, live).restoreEntities(vcards, addressBookUid);
				}
			}
		} catch (Exception e) {
			logger.error("Error while restoring addressbook {}@{}: {}", restorableItem.entryUid(),
					restorableItem.domain(), e.getMessage());
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}

}
