/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.sdsspool;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

import com.github.luben.zstd.RecyclingBufferPool;
import com.github.luben.zstd.ZstdInputStream;

import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.ExistResponse;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.MgetRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.dto.TierMoveRequest;
import net.bluemind.sds.dto.TierMoveResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.PathHandler;

public class SdsDataProtectSpool implements ISdsSyncStore {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/sds-spool/spool");
	private final Path root;

	public SdsDataProtectSpool() {
		this(DEFAULT_PATH);
	}

	public SdsDataProtectSpool(Path root) {
		this.root = root;
	}

	public Path path() {
		return root;
	}

	public Path livePath(String guid) {
		return root.resolve(String.valueOf(guid.charAt(0))).resolve(String.valueOf(guid.charAt(1)))
				.resolve(guid + ".zst");
	}

	private void get(String guid, OutputStream out) throws IOException {
		Path inpath = livePath(guid);
		try (InputStream rawin = Files.newInputStream(inpath);
				ZstdInputStream in = new ZstdInputStream(rawin, RecyclingBufferPool.INSTANCE)) {
			in.transferTo(out);
		}
	}

	private void getRaw(String guid, OutputStream out) throws IOException {
		Path inpath = livePath(guid);
		try (InputStream rawin = Files.newInputStream(inpath)) {
			rawin.transferTo(out);
		}
	}

	@Override
	public ExistResponse exists(ExistRequest req) {
		return ExistResponse.from(Files.exists(livePath(req.guid)));
	}

	@Override
	public SdsResponse upload(PutRequest req) {
		throw new UnsupportedOperationException();
	}

	@Override
	public SdsResponse download(GetRequest req) {

		try (var out = PathHandler.forPathOrUri(req.filename).openForWriting()) {
			get(req.guid, out);
		} catch (IOException ie) {
			return SdsResponse.error("Local IO error " + ie.getMessage() + " on " + req, req.guid, true);
		} catch (Exception e) {
			return SdsResponse.error("Error " + e.getMessage() + " on " + req, req.guid, true);
		}
		return SdsResponse.UNTAGGED_OK;
	}

	@Override
	public SdsResponse downloadRaw(GetRequest req) {
		try (var out = PathHandler.forPathOrUri(req.filename).openForWriting()) {
			getRaw(req.guid, out);
		} catch (IOException ie) {
			return SdsResponse.error("Local IO error " + ie.getMessage() + " on " + req, req.guid, true);
		} catch (Exception e) {
			return SdsResponse.error("Error " + e.getMessage() + " on " + req, req.guid, true);
		}
		return SdsResponse.UNTAGGED_OK;
	}

	@Override
	public SdsResponse downloads(MgetRequest mget) {
		List<SdsResponse> errors = mget.transfers.stream().map(tx -> GetRequest.of(mget.mailbox, tx.guid, tx.filename))
				.parallel().map(mget.raw ? this::downloadRaw : this::download).filter(resp -> !resp.succeeded())
				.toList();
		if (!errors.isEmpty()) {
			var errMsg = new StringBuilder();
			for (var err : errors) {
				errMsg.append(err.error.toString());
			}
			List<String> guids = errors.stream().map(e -> e.error.guids()).flatMap(List::stream).toList();
			return SdsResponse.error(errMsg.toString(), guids, true);
		}
		return SdsResponse.UNTAGGED_OK;
	}

	@Override
	public SdsResponse delete(DeleteRequest req) {
		throw new UnsupportedOperationException();
	}

	@Override
	public TierMoveResponse tierMove(TierMoveRequest tierMoveRequest) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void close() {
		// ok
	}
}
