/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.calendar.pdf.internal;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.ResourceBundle;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.w3c.dom.Document;
import org.w3c.tidy.Tidy;
import org.xml.sax.InputSource;

import com.itextpdf.html2pdf.ConverterProperties;
import com.itextpdf.html2pdf.HtmlConverter;

import freemarker.ext.beans.BeansWrapper;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import net.bluemind.calendar.api.PrintOptions;
import net.bluemind.calendar.api.PrintOptions.CalendarMetadata;
import net.bluemind.calendar.api.VEvent;
import net.bluemind.calendar.pdf.PrintCalendarHelper;
import net.bluemind.calendar.pdf.internal.imageio.ImageIOUtil;
import net.bluemind.core.api.date.BmDateTimeWrapper;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemContainerValue;
import net.bluemind.icalendar.api.ICalendarElement.Attendee;
import net.bluemind.icalendar.api.ICalendarElement.Organizer;
import net.bluemind.icalendar.api.ICalendarElement.Role;

public class PrintCalendarList extends PrintCalendar {

	private static Configuration cfg;

	public static class CalDisplay {
		public String name;
		public String color;

		public String getName() {
			return name;
		}

		public String getColor() {
			return color;
		}

		public CalDisplay(String name, String color) {
			this.name = name;
			this.color = color;
		}

	}

	public static class CalDay {
		public String name;
		public List<PrintEvent> events;

		public CalDay(String name, List<PrintEvent> events) {
			super();
			this.name = name;
			this.events = events;
		}

		public String getName() {
			return name;
		}

		public List<PrintEvent> getEvents() {
			return events;
		}

	}

	public static class PrintEvent {
		public String timeSlot;
		public VEvent event;
		public String title;
		public String color;
		public String textDecoration;
		public boolean details;
		public List<CalDisplay> attendees;
		private List<CalDisplay> fattendees;
		private CalDisplay chair;
		private String description;

		public PrintEvent(String timeSlot, String color, String textDecoration, String title, boolean details,
				VEvent event, CalDisplay chair, List<CalDisplay> attendees, List<CalDisplay> fattendees,
				String description) {
			super();
			this.timeSlot = timeSlot;
			this.event = event;
			this.title = title;
			this.color = color;
			this.textDecoration = textDecoration;
			this.details = details;
			this.chair = chair;
			this.attendees = attendees;
			this.fattendees = fattendees;
			this.description = description;
		}

		public String getTimeSlot() {
			return timeSlot;
		}

		public VEvent getEvent() {
			return event;
		}

		public String getTitle() {
			return title;
		}

		public boolean isDetails() {
			return details;
		}

		public List<CalDisplay> getAttendees() {
			return attendees;
		}

		public List<CalDisplay> getFattendees() {
			return fattendees;
		}

		public CalDisplay getChair() {
			return chair;
		}

		public String getDescription() {
			return description;
		}
	}

	static {
		cfg = new Configuration(Configuration.DEFAULT_INCOMPATIBLE_IMPROVEMENTS);

		cfg.setClassForTemplateLoading(PrintCalendarList.class, "/tpl");
		cfg.setTagSyntax(Configuration.AUTO_DETECT_TAG_SYNTAX);
		BeansWrapper wrapper = new BeansWrapper(Configuration.DEFAULT_INCOMPATIBLE_IMPROVEMENTS);
		wrapper.setExposeFields(true);
		cfg.setObjectWrapper(wrapper);
	}

	private String htmlDocument;
	private Map<Long, List<ItemContainerValue<VEvent>>> ocs;
	private ResourceBundle rb;

	public PrintCalendarList(PrintContext context, PrintOptions options, List<ItemContainerValue<VEvent>> vevents)
			throws ServerFault {
		super(context, options);

		addPage();
		ocs = sortOccurrences(vevents, options.dateBegin, options.dateEnd);
		rb = ResourceBundle.getBundle("lang/translation", l);

	}

	@Override
	public void process() throws ServerFault {
		Template tpl = null;
		try {
			tpl = cfg.getTemplate("list-print.ftl");
		} catch (IOException e) {
			throw new ServerFault(e);
		}

		Map<String, Object> model = new HashMap<>();
		model.put("msg", new MessageResolverMethod(rb, l));

		Calendar c = Calendar.getInstance(GMTtimezone);
		c.setTimeZone(timezone);
		Date now = c.getTime();

		SimpleDateFormat footerDateFormat = new SimpleDateFormat(
				userSettings.get("date") + ", " + userSettings.get("timeformat"), l);

		footerDateFormat.setTimeZone(timezone);
		model.put("printDate", footerDateFormat.format(now));

		Calendar start = Calendar.getInstance(timezone);
		start.setTimeInMillis(new BmDateTimeWrapper(options.dateBegin).toTimestamp(timezone.getID()));
		// not done in other print
		// need it here ?
		// FIXME question
		// options.dateBegin + start.get(Calendar.ZONE_OFFSET) +
		// start.get(Calendar.DST_OFFSET));

		Calendar end = Calendar.getInstance(timezone);
		end.setTimeInMillis(new BmDateTimeWrapper(options.dateEnd).toTimestamp(timezone.getID()));
		// FIXME question same as start
		// end.setTimeInMillis(options.getDateEnd() - 1 +
		// end.get(Calendar.ZONE_OFFSET) + end.get(Calendar.DST_OFFSET));

		String date = dateFormat.format(start.getTime());
		if (!date.equals(dateFormat.format(end.getTime()))) {
			date += " - " + dateFormat.format(end.getTime());
		}

		model.put("date", date);

		List<CalDisplay> cals = new ArrayList<>();
		for (CalendarMetadata cal : options.calendars) {
			CalInfo calInfo = calInfos.get(cal.uid);
			cals.add(new CalDisplay(calInfo.name, calInfo.color));
		}
		model.put("cals", cals);
		model.put("options", options);

		model.put("multi", cals.size() > 1);

		addEvents(model);
		StringWriter sw = new StringWriter();
		try {
			tpl.process(model, sw);
		} catch (TemplateException | IOException e) {
			throw new ServerFault(e);
		}

		htmlDocument = sw.toString();
	}

	private void addEvents(Map<String, Object> model) {
		List<CalDay> days = new ArrayList<>();
		for (Entry<Long, List<ItemContainerValue<VEvent>>> entry : ocs.entrySet()) {
			Calendar c = Calendar.getInstance(timezone);
			c.setTimeInMillis(entry.getKey());

			List<ItemContainerValue<VEvent>> occurrences = entry.getValue();
			Collections.sort(occurrences, new EventComparator(timezone.getID()));

			List<PrintEvent> dayEvents = new ArrayList<>();
			for (ItemContainerValue<VEvent> o : occurrences) {
				PrintEvent pe = addEvent(o);
				dayEvents.add(pe);
			}

			CalDay cd = new CalDay(dateFormat.format(c.getTime()), dayEvents);
			days.add(cd);
		}

		model.put("days", days);
	}

	protected PrintEvent addEvent(ItemContainerValue<VEvent> item) {
		VEvent e = item.value;

		List<Attendee> attendees = new ArrayList<>(e.attendees);

		boolean priv = isPrivateEvent(e);

		String title = getTitle(e, priv);
		String color = this.getCalInfo(item.containerUid).color;
		String textDecoration = PrintCalendarHelper.isDeclinedOrCanceledEvent(
				PrintCalendarHelper.getPart(e, item.containerUid), e.status) ? "line-through" : "none";
		String timeSlot = null;
		if (e.allDay()) {
			timeSlot = lang.getProperty("allday");
		} else {

			Calendar begin = Calendar.getInstance(timezone);
			begin.setTimeInMillis(new BmDateTimeWrapper(e.dtstart).toTimestamp(timezone.getID()));
			Calendar end = (Calendar) begin.clone();
			end.add(Calendar.SECOND, duration(e).intValue());
			timeSlot = timeFormat.format(begin.getTime()) + "-" + timeFormat.format(end.getTime());
		}

		List<CalDisplay> atts = new ArrayList<>();
		List<CalDisplay> fatts = new ArrayList<>();

		Attendee organizerToChair = organizerToChair(item.value.organizer);
		if (organizerToChair != null) {
			attendees.add(organizerToChair);
		}
		if (attendees.isEmpty()) {
			attendees.add(calendarToChair(this.getCalInfo(item.containerUid)));
		}
		CalDisplay chair = null;
		for (Attendee a : attendees) {
			if (securityContext.getSubject().equals(a.uri) // FIXME
					|| !priv) {
				String attendeeColor = null;
				CalInfo calInfo = getCalInfo(a.uri);
				if (calInfo != null) {
					attendeeColor = calInfo.color;
				}
				String name = a.commonName;
				if (name == null) {
					name = a.mailto;
				}
				CalDisplay calDisplay = new CalDisplay(name, attendeeColor);

				if (a.role != null) {
					switch (a.role) {
					case RequiredParticipant:
						atts.add(calDisplay);
						if (chair == null) {
							chair = calDisplay;
						}
						break;
					case OptionalParticipant:
						fatts.add(calDisplay);
						break;
					case Chair:
						chair = calDisplay;
						break;
					default:
						break;

					}
				}
			}
		}

		String description = null;
		if (!priv && e.description != null && e.description.length() > 0) {
			StringWriter dw = new StringWriter();
			Tidy tidy = new Tidy();
			tidy.setInputEncoding("UTF-8");
			tidy.setOutputEncoding("UTF-8");
			tidy.setWraplen(Integer.MAX_VALUE);
			tidy.setPrintBodyOnly(true);
			tidy.setXmlOut(true);
			tidy.setSmartIndent(true);
			tidy.parseDOM(new StringReader(e.description), dw);
			description = dw.toString();

		}

		return new PrintEvent(timeSlot, color, textDecoration, title, options.showDetail, e, chair, atts, fatts,
				description);
	}

	private Attendee organizerToChair(Organizer organizer) {
		if (organizer == null) {
			return null;
		}
		Attendee attendee = new Attendee();
		attendee.role = Role.Chair;
		attendee.commonName = organizer.commonName;
		attendee.uri = organizer.uri;
		if (attendee.uri == null || attendee.uri.isEmpty()) {
			attendee.uri = organizer.dir;
		}
		attendee.mailto = organizer.mailto;
		return attendee;
	}

	private Attendee calendarToChair(CalInfo calendar) {
		Attendee attendee = new Attendee();
		attendee.role = Role.Chair;
		attendee.commonName = calendar.name;
		attendee.uri = calendar.uid;
		return attendee;
	}

	private CalInfo getCalInfo(String uri) {
		if (uri == null) {
			return null;
		}
		if (uri.contains("/")) {
			uri = uri.substring(uri.lastIndexOf('/') + 1);
		}
		CalInfo calInfo = calInfos.get(uri);
		if (null == calInfo) {
			calInfo = calInfos.get("calendar:" + uri);
			if (null == calInfo) {
				calInfo = calInfos.get("calendar:Default:" + uri);
			}
		}
		return calInfo;
	}

	@Override
	public byte[] sendPNGString() {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		ConverterProperties converterProperties = new ConverterProperties();
		ByteArrayInputStream pdfIn = new ByteArrayInputStream(htmlDocument.getBytes());

		try {
			HtmlConverter.convertToPdf(pdfIn, os, converterProperties);
		} catch (IOException e) {
			try {
				os.close();
			} catch (IOException e1) {
			}
			throw new ServerFault(e);
		}

		ByteArrayOutputStream ios = new ByteArrayOutputStream();
		try {
			PDDocument document = Loader.loadPDF(os.toByteArray());
			PDFRenderer pdfRenderer = new PDFRenderer(document);
			BufferedImage bim = pdfRenderer.renderImageWithDPI(0, 300, ImageType.RGB);
			ImageIOUtil.writeImage(bim, "PNG", ios, 300);
		} catch (IOException e) {
			throw new ServerFault(e);
		}

		return ios.toByteArray();
	}

	@Override
	public byte[] sendPDFString() throws ServerFault {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		ConverterProperties converterProperties = new ConverterProperties();
		ByteArrayInputStream pdfIn = new ByteArrayInputStream(htmlDocument.getBytes());
		try {
			HtmlConverter.convertToPdf(pdfIn, os, converterProperties);
		} catch (IOException e) {
			try {
				os.close();
			} catch (IOException e1) {
			}
			throw new ServerFault(e);
		}
		return os.toByteArray();
	}

	@Override
	public byte[] sendSVGString() throws ServerFault {
		byte[] ret = null;
		try {

			DocumentBuilder db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			InputSource is = new InputSource();
			is.setCharacterStream(new StringReader(htmlDocument));
			Document doc = db.parse(is);

			ByteArrayOutputStream out = new ByteArrayOutputStream();
			TransformerFactory fac = TransformerFactory.newInstance();
			Transformer tf = fac.newTransformer();
			tf.setOutputProperty(OutputKeys.INDENT, "no");
			Source input = new DOMSource(doc.getParentNode());
			Result output = new StreamResult(out);
			tf.transform(input, output);
			ret = out.toByteArray();
		} catch (Exception t) {
			throw new ServerFault(t.getMessage(), t);
		}
		return ret;
	}

}
