/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.deserializer;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.dataprotect.service.tool.CoreCommonBackupWorker;

public class CyrusSdsBackupZipMailbox extends CyrusSdsBackupMailbox {
	protected static Logger logger = LoggerFactory.getLogger(CyrusSdsBackupZipMailbox.class);

	public CyrusSdsBackupZipMailbox(MailboxIndexJson mailboxIndex, Path workingFolder) throws Exception {
		super(mailboxIndex, workingFolder);
	}

	private Path getZipFilePath() throws IOException {
		if (!Files.isDirectory(storageFolder)) {
			return null;
		}

		Path indexPath = storageFolder.resolve("index.json");
		if (!Files.exists(indexPath)) {
			return null;
		}

		MailboxIndexJsonList indexJson = CoreCommonBackupWorker.readIndexJson(storageFolder);
		if (indexJson == null) {
			throw new FileNotFoundException("index.json File " + indexPath + " not found");
		}

		Optional<String> byOwnerUid = indexJson.mailboxesIndexInfo().stream()
				.filter(i -> i.mailboxUid().equals(mailboxIndex.mailboxUid()) && i.zipFileName() != null)
				.map(i -> i.zipFileName()).findFirst();

		if (byOwnerUid.isEmpty()) {
			return null;
		}
		return storageFolder.resolve(byOwnerUid.get());
	}

	private ByteBuf readZipFile() throws IOException {
		Path zipFilePath = getZipFilePath();
		if (zipFilePath == null) {
			return null;
		}

		return JsonReadWrite.findFilesInZipFile(zipFilePath, mailboxIndex.filename());
	}

	@Override
	protected JsonParser readJsonFileContent() throws IOException {
		ByteBuf jsonContent = null;
		JsonFactory jsonfactory = new JsonFactory();

		if (mailboxIndex.mailboxUid() != null) {
			jsonContent = readZipFile();
			if (jsonContent != null) {
				return jsonfactory.createParser((InputStream) new ByteBufInputStream(jsonContent));
			}
		}

		return jsonfactory.createParser(storageFolder.resolve(mailboxIndex.filename()).toFile());
	}

}