/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.sds.store.s3;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.util.concurrent.CompletableFuture;

import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import io.vertx.core.Vertx;
import net.bluemind.sds.store.PathHandler;
import software.amazon.awssdk.core.async.SdkPublisher;

public class PathResponseTransformer<T> implements IResponseTransformer<T> {

	private final PathHandler path;
	private CompletableFuture<T> cf;
	private T response;
	private long transferred;

	public PathResponseTransformer(Vertx vertx, String path) {
		this.path = PathHandler.forPathOrUri(path);
	}

	@Override
	public CompletableFuture<T> prepare() {
		cf = new CompletableFuture<>();
		return cf.thenApply(v -> response);
	}

	@Override
	public void onResponse(T response) {
		this.response = response;
	}

	@Override
	public void onStream(SdkPublisher<ByteBuffer> publisher) {

		publisher.subscribe(new Subscriber<ByteBuffer>() {

			private Subscription sub;
			private SeekableByteChannel out;

			@Override
			public void onSubscribe(Subscription s) {
				this.sub = s;
				try {
					out = path.openWriteChannel();
					s.request(1);
				} catch (IOException e) {
					exceptionOccurred(e);
				}
			}

			@Override
			public void onNext(ByteBuffer t) {
				try {
					transferred += out.write(t);
					sub.request(1);
				} catch (IOException e) {
					onError(e);
				}
			}

			@Override
			public void onError(Throwable t) {
				exceptionOccurred(t);
			}

			@Override
			public void onComplete() {
				try {
					out.close();
				} catch (IOException e) {
					// ok
				}
				cf.complete(null);
			}
		});

	}

	@Override
	public void exceptionOccurred(Throwable error) {
		path.cleanup();
		cf.completeExceptionally(error);
	}

	@Override
	public long transferred() {
		return transferred;
	}

}
