/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.storage.jdbc;

import java.time.DateTimeException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.hash.Hashing;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.calendar.api.ICalendar;
import net.bluemind.common.task.Tasks;
import net.bluemind.config.Token;
import net.bluemind.core.container.api.ContainerHierarchyNode;
import net.bluemind.core.container.api.ContainerSubscription;
import net.bluemind.core.container.api.ContainerSubscriptionModel;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.IContainersFlatHierarchy;
import net.bluemind.core.container.api.IOwnerSubscriptions;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ContainerModifiableDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.http.ClientSideServiceProvider;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.device.api.IDevice;
import net.bluemind.device.api.IDevices;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.eas.api.Account;
import net.bluemind.eas.api.FolderSyncVersions;
import net.bluemind.eas.api.Heartbeat;
import net.bluemind.eas.api.IEas;
import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.MailFolder;
import net.bluemind.eas.backend.MailboxVacation;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.HierarchyImportFolderUpdateEntity;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.device.DeviceId;
import net.bluemind.eas.dto.settings.OofMessage;
import net.bluemind.eas.dto.settings.OofMessage.Audience;
import net.bluemind.eas.dto.settings.OofState;
import net.bluemind.eas.dto.settings.SettingsRequest.Oof;
import net.bluemind.eas.dto.settings.SettingsResponse;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.dto.user.MSUser;
import net.bluemind.eas.exception.CollectionNotFoundException;
import net.bluemind.eas.exception.ServerErrorException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.MailFilter.Vacation;
import net.bluemind.network.topology.Topology;
import net.bluemind.todolist.api.ITodoList;
import net.bluemind.user.api.IUserSubscription;

/**
 * Store device infos, id mappings & last sync dates into BM database
 *
 */
public class SyncStorage implements ISyncStorage {
	private static final Logger logger = LoggerFactory.getLogger(SyncStorage.class);

	/**
	 * Keep at most 64MB of search results mapping and with a 10min TTL for each
	 * 
	 * Keeping them as TTL keys in redis would be better to survive EAS server
	 * restart.
	 */
	private static final Cache<Long, byte[]> ephemeralLongIdMapping = Caffeine.newBuilder().weigher((k, v) -> 64)
			.maximumWeight(64_000_000).expireAfterWrite(10, TimeUnit.MINUTES).build();

	private String locateCore() {
		return Topology.getIfAvailable().map(t -> "http://" + t.core().value.address() + ":8090")
				.orElse("http://127.0.0.1:8090");
	}

	private IServiceProvider admin0Provider() {
		return ClientSideServiceProvider.getProvider(locateCore(), Token.admin0()).setOrigin("bm-eas-syncStorage");
	}

	private IServiceProvider provider(BackendSession bs) {
		return ClientSideServiceProvider.getProvider(locateCore(), bs.getSid())
				.setOrigin("bm-eas-syncStorage-" + bs.getUniqueIdentifier());
	}

	// SystemConf
	@Override
	public String getSystemConf(String key) {
		return Shared.mapSysconf().get(key);
	}

	@Override
	public Map<String, String> getSystemConf() {
		return Shared.mapSysconf().asMap();
	}

	// Device/Auth stuff
	@Override
	public Map<String, String> getWipedDevices() {
		try {
			IDevices service = admin0Provider().instance(IDevices.class);
			return service.listWiped().stream()
					.collect(Collectors.toMap(entry -> entry.identifier, entry -> entry.wipeMode.name()));
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return Collections.emptyMap();
		}
	}

	@Override
	public void updateLastSync(BackendSession bs) {
		try {
			IDevice deviceService = admin0Provider().instance(IDevice.class, bs.getUser().getUid());
			deviceService.updateLastSync(bs.getDeviceId().getInternalId());
		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
		}
	}

	// Heartbeat stuff
	@Override
	public long findLastHeartbeat(DeviceId deviceId) {
		try {
			Heartbeat heartbeat = getService().getHeartbeat(deviceId.getIdentifier());
			if (heartbeat != null) {
				return heartbeat.value;
			}

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		return 0L;
	}

	@Override
	public synchronized void updateLastHearbeat(DeviceId deviceId, long value) {
		Heartbeat heartbeat = new Heartbeat();
		heartbeat.deviceUid = deviceId.getIdentifier();
		heartbeat.value = value;
		try {
			getService().setHeartbeat(heartbeat);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	public HierarchyNode getHierarchyNode(String origin, String domainUid, String userUid, String nodeUid)
			throws CollectionNotFoundException {
		ItemValue<ContainerHierarchyNode> folder = getFolderByHierarchyNode(origin, domainUid, userUid,
				service -> service.getComplete(nodeUid));

		if (folder == null) {
			throw new CollectionNotFoundException("Container " + nodeUid + " not found");
		}

		return new HierarchyNode(CollectionId.of(Long.toString(folder.internalId)), folder.value.containerUid,
				folder.value.containerType);
	}

	@Override
	public HierarchyNode getHierarchyNode(CollectionIdContext collectionIdContext) throws CollectionNotFoundException {
		if (collectionIdContext.collectionId().getSubscriptionId().isPresent()) {
			IOwnerSubscriptions subs = provider(collectionIdContext.backendSession()).instance(
					IOwnerSubscriptions.class, collectionIdContext.backendSession().getUser().getDomain(),
					collectionIdContext.backendSession().getUser().getUid());
			ItemValue<ContainerSubscriptionModel> subscription = subs
					.getCompleteById(collectionIdContext.collectionId().getSubscriptionId().get());
			if (subscription == null || subscription.value == null) {
				throw new CollectionNotFoundException("Failed to get hierarchy node. Invalid subscription "
						+ collectionIdContext.collectionId().getSubscriptionId().get());
			}
			return getHierarchyNode(collectionIdContext, subscription.value.owner);
		}
		return getHierarchyNode(collectionIdContext, collectionIdContext.backendSession().getUser().getUid());

	}

	private HierarchyNode getHierarchyNode(CollectionIdContext collectionIdContext, String owner)
			throws CollectionNotFoundException {

		// FIXME
		if (owner == null) {
			owner = collectionIdContext.backendSession().getUser().getUid();
		}

		ItemValue<ContainerHierarchyNode> folder = getFolderByHierarchyNode(
				collectionIdContext.backendSession().getUniqueIdentifier(),
				collectionIdContext.backendSession().getUser().getDomain(), owner,
				service -> service.getCompleteById(collectionIdContext.collectionId().getFolderId()));

		if (folder == null) {
			throw new CollectionNotFoundException(
					"Collection " + owner + ":" + collectionIdContext.collectionId() + " not found");
		}

		long subscriptionId = collectionIdContext.collectionId().getSubscriptionId().isPresent()
				? collectionIdContext.collectionId().getSubscriptionId().get()
				: 0;
		return new HierarchyNode(CollectionId.of(subscriptionId, Long.toString(folder.internalId)),
				folder.value.containerUid, folder.value.containerType);
	}

	private ItemValue<ContainerHierarchyNode> getFolderByHierarchyNode(String origin, String domain, String owner,
			Function<IContainersFlatHierarchy, ItemValue<ContainerHierarchyNode>> nodeGetFunc) {
		ItemValue<ContainerHierarchyNode> folder = null;
		try {
			folder = getFolderInUserHierarchy(origin, domain, owner, nodeGetFunc);
		} catch (Exception e) {
			// hierarchy container might not exist
		}
		if (folder == null) {
			folder = getFolderInDomainHierarchy(origin, domain, nodeGetFunc);
		}
		return folder;
	}

	private ItemValue<ContainerHierarchyNode> getFolderInUserHierarchy(String origin, String domain, String owner,
			Function<IContainersFlatHierarchy, ItemValue<ContainerHierarchyNode>> nodeGetFunc) {
		return nodeGetFunc.apply(getIContainersFlatHierarchyService(origin, domain, owner));
	}

	private ItemValue<ContainerHierarchyNode> getFolderInDomainHierarchy(String origin, String domain,
			Function<IContainersFlatHierarchy, ItemValue<ContainerHierarchyNode>> nodeGetFunc) {
		return nodeGetFunc.apply(getIContainersFlatHierarchyService(origin, domain,
				UUID.nameUUIDFromBytes(domain.getBytes()).toString()));
	}

	@Override
	public MailFolder getMailFolder(CollectionIdContext collectionIdContext) throws CollectionNotFoundException {
		HierarchyNode folder = getHierarchyNode(collectionIdContext);
		String uniqueId = IMailReplicaUids.uniqueId(folder.containerUid);
		ItemValue<MailboxFolder> mailFolder = getIMailboxFoldersService(collectionIdContext).getComplete(uniqueId);

		if (mailFolder == null) {
			throw new CollectionNotFoundException(
					String.format("MailboxFolder CollectionId[%s], FolderId:[%s] not found",
							collectionIdContext.collectionId(), uniqueId));
		}

		return new MailFolder(collectionIdContext.collectionId(), mailFolder.uid, mailFolder.value.name,
				mailFolder.value.fullName, mailFolder.value.parentUid);
	}

	@Override
	public MailFolder getMailFolderByName(BackendSession bs, String name) throws CollectionNotFoundException {
		ItemValue<MailboxFolder> mailFolder = getIMailboxFoldersService(bs).byName(name);
		if (mailFolder == null) {
			throw new CollectionNotFoundException("MailboxFolder CollectionId[" + name + "] not found");
		}
		MSUser user = bs.getUser();
		String cont = IMailReplicaUids.mboxRecords(mailFolder.uid);
		String hNodeUid = ContainerHierarchyNode.uidFor(cont, IMailReplicaUids.MAILBOX_RECORDS, user.getDomain());
		ItemValue<ContainerHierarchyNode> hNode = getIContainersFlatHierarchyService(bs.getUniqueIdentifier(),
				user.getDomain(), user.getUid()).getComplete(hNodeUid);
		if (hNode == null) {
			throw new CollectionNotFoundException(
					"MailboxFolder CollectionId[" + name + "], FolderId:[" + hNodeUid + "] not found");
		}
		return new MailFolder(CollectionId.of(Long.toString(hNode.internalId)), mailFolder.uid, mailFolder.value.name,
				mailFolder.value.fullName, mailFolder.value.parentUid);
	}

	@Override
	public CollectionId createFolder(HierarchyImportFolderUpdateEntity hierarchyEntity) {
		Long folderUid = null;
		try {
			String uid = UUID.randomUUID().toString();
			MSUser user = hierarchyEntity.backendSession.getUser();
			if (hierarchyEntity.type == ItemDataType.TASKS || hierarchyEntity.type == ItemDataType.CALENDAR) {
				ContainerDescriptor descriptor = null;
				String containerNodeUid = null;
				if (hierarchyEntity.type == ItemDataType.TASKS) {
					descriptor = ContainerDescriptor.create(uid, hierarchyEntity.folderDisplayName, user.getUid(),
							"todolist", user.getDomain(), false);
					containerNodeUid = ContainerHierarchyNode.uidFor(uid, "todolist", user.getDomain());

				} else {
					descriptor = ContainerDescriptor.create(uid, hierarchyEntity.folderDisplayName, user.getUid(),
							"calendar", user.getDomain(), false);
					containerNodeUid = ContainerHierarchyNode.uidFor(uid, "calendar", user.getDomain());

				}

				IContainers containers = admin0Provider().instance(IContainers.class);
				containers.create(uid, descriptor);
				IContainerManagement manager = admin0Provider().instance(IContainerManagement.class, uid);
				manager.setAccessControlList(Arrays.asList(AccessControlEntry.create(user.getUid(), Verb.Write)));

				IUserSubscription userSubService = admin0Provider().instance(IUserSubscription.class, user.getDomain());
				userSubService.subscribe(user.getUid(), Arrays.asList(ContainerSubscription.create(uid, false)));

				ItemValue<ContainerHierarchyNode> folder = getIContainersFlatHierarchyService(
						hierarchyEntity.backendSession.getUniqueIdentifier(), user.getDomain(), user.getUid())
						.getComplete(containerNodeUid);

				folderUid = folder.internalId;
			}

		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(hierarchyEntity.user, e, logger);
		}

		return CollectionId.of(Long.toString(folderUid));

	}

	@Override
	public boolean deleteFolder(HierarchyImportFolderUpdateEntity hierarchyEntity) {
		boolean ret = false;
		IServiceProvider cssp = provider(hierarchyEntity.backendSession);

		IContainers containers = cssp.instance(IContainers.class);

		try {
			BaseContainerDescriptor container = containers.getLight(hierarchyEntity.node.containerUid);

			if (hierarchyEntity.type == ItemDataType.TASKS) {
				if (container.defaultContainer) {
					EasLogUser.logWarnAsUser(hierarchyEntity.user, logger, "Cannot delete default todolist {}",
							hierarchyEntity.node.containerUid);
					return true;
				}

				ITodoList service = cssp.instance(ITodoList.class, hierarchyEntity.node.containerUid);
				// ITodoList.reset should return a TaskRef.
				// see BM-13104
				service.reset();
			} else if (hierarchyEntity.type == ItemDataType.CALENDAR) {
				if (container.defaultContainer) {
					EasLogUser.logWarnAsUser(hierarchyEntity.user, logger, "Cannot delete default calendar {}",
							hierarchyEntity.node.containerUid);
					return true;
				}

				ICalendar service = cssp.instance(ICalendar.class, hierarchyEntity.node.containerUid);
				TaskRef tr = service.reset();
				Tasks.followStream(cssp, logger, null, tr);
			}

			containers.delete(hierarchyEntity.node.containerUid);

			ret = true;
		} catch (Exception e) {
			EasLogUser.logErrorExceptionAsUser(hierarchyEntity.user, e, logger, "Fail to delete folder {}",
					hierarchyEntity.node.containerUid);
		}
		return ret;
	}

	@Override
	public boolean updateFolder(HierarchyImportFolderUpdateEntity hierarchyEntity) {
		boolean ret = false;

		try {
			IContainers containers = provider(hierarchyEntity.backendSession).instance(IContainers.class);
			if (hierarchyEntity.type == ItemDataType.CALENDAR || hierarchyEntity.type == ItemDataType.TASKS) {
				BaseContainerDescriptor container = containers.getLight(hierarchyEntity.node.containerUid);
				if (container.defaultContainer) {
					EasLogUser.logWarnAsUser(hierarchyEntity.user, logger, "Cannot update default {} {}",
							hierarchyEntity.type, hierarchyEntity.node.containerUid);
					return true;
				}
			}

			ContainerModifiableDescriptor descriptor = new ContainerModifiableDescriptor();
			descriptor.name = hierarchyEntity.folderDisplayName;
			descriptor.defaultContainer = false;

			containers.update(hierarchyEntity.node.containerUid, descriptor);

			ret = true;
		} catch (Exception e) {
			EasLogUser.logErrorExceptionAsUser(hierarchyEntity.user, e, logger, "Fail to update folder {}",
					hierarchyEntity.node.containerUid);
		}
		return ret;
	}

	// FolderState
	@Override
	public boolean needReset(BackendSession bs) {
		boolean ret = false;
		try {
			ret = getService().needReset(Account.create(bs.getUser().getUid(), bs.getDeviceId().getIdentifier()));

		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
		}
		return ret;
	}

	@Override
	public void resetFolder(BackendSession bs) {
		try {
			getService().deletePendingReset(Account.create(bs.getUser().getUid(), bs.getDeviceId().getIdentifier()));

		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
		}

	}

	@Override
	public void insertClientId(String clientId) {
		try {
			getService().insertClientId(clientId);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	public boolean isKnownClientId(String clientId) {
		boolean ret = true;
		try {
			ret = getService().isKnownClientId(clientId);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
		return ret;
	}

	@Override
	public void setFolderSyncVersions(FolderSyncVersions versions) {
		try {
			getService().setFolderSyncVersions(versions);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

	}

	@Override
	public Map<String, String> getFolderSyncVersions(Account account) {
		try {
			return getService().getFolderSyncVersions(account);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
		return null;
	}

	/**
	 * @param authKey
	 * @return
	 */
	private IEas getService() {
		return admin0Provider().instance(IEas.class);
	}

	private IContainersFlatHierarchy getIContainersFlatHierarchyService(String origin, String domainUid, String owner) {
		return ClientSideServiceProvider.getProvider(locateCore(), Token.admin0())
				.setOrigin("bm-eas-syncStorage-" + origin).instance(IContainersFlatHierarchy.class, domainUid, owner);
	}

	private IMailboxFolders getIMailboxFoldersService(CollectionIdContext collectionIdContext) {
		CyrusPartition part = CyrusPartition.forServerAndDomain(
				collectionIdContext.backendSession().getUser().getDataLocation(),
				collectionIdContext.backendSession().getUser().getDomain());
		String mailboxRoot = "user." + collectionIdContext.backendSession().getUser().getUid().replace('.', '^');
		if (collectionIdContext.collectionId().getSubscriptionId().isPresent()) {
			IOwnerSubscriptions subscriptionsService = provider(collectionIdContext.backendSession()).instance(
					IOwnerSubscriptions.class, collectionIdContext.backendSession().getUser().getDomain(),
					collectionIdContext.backendSession().getUser().getUid());
			ItemValue<ContainerSubscriptionModel> subscription = subscriptionsService
					.getCompleteById(collectionIdContext.collectionId().getSubscriptionId().get());

			IDirectory directoryService = admin0Provider().instance(IDirectory.class,
					collectionIdContext.backendSession().getUser().getDomain());
			DirEntry dirEntry = directoryService.findByEntryUid(subscription.value.owner);
			if (dirEntry.kind == Kind.USER || dirEntry.kind == Kind.SHARED_MAILBOX) {
				mailboxRoot = "user." + dirEntry.entryUid.replace('.', '^');
			} else {
				mailboxRoot = dirEntry.entryUid.replace('.', '^');
			}
			part = CyrusPartition.forServerAndDomain(dirEntry.dataLocation,
					collectionIdContext.backendSession().getUser().getDomain());
		}

		return provider(collectionIdContext.backendSession()).instance(IMailboxFolders.class, part.name, mailboxRoot);
	}

	private IMailboxFolders getIMailboxFoldersService(BackendSession bs) {
		CyrusPartition part = CyrusPartition.forServerAndDomain(bs.getUser().getDataLocation(),
				bs.getUser().getDomain());
		// FIXME
		// FIXME
		// FIXME
		// FIXME
		// FIXME
		return provider(bs).instance(IMailboxFolders.class, part.name,
				"user." + bs.getUser().getUid().replace('.', '^'));
	}

	@Override
	public MailboxVacation getVacationForMailbox(BackendSession bs) {
		IMailboxes mailboxesService = admin0Provider().instance(IMailboxes.class, bs.getUser().getDomain());
		Vacation v = mailboxesService.getMailboxVacation(bs.getUser().getUid());
		return new MailboxVacation(v.enabled, v.start, v.end, v.text, v.textHtml, v.subject);
	}

	@Override
	public void setVacationForMailbox(BackendSession bs, Oof.Set set) throws ServerErrorException {
		Vacation vacation = fromSet(set);
		try {
			IMailboxes mailboxesService = admin0Provider().instance(IMailboxes.class, bs.getUser().getDomain());
			mailboxesService.setMailboxVacation(bs.getUser().getUid(), vacation);
		} catch (Exception e) {
			throw new ServerErrorException(e);
		}
	}

	@Override
	public boolean userHasWriteAccess(BackendSession bs, String mboxRecordContainerUid) {
		IContainers containerService = provider(bs).instance(IContainers.class);
		ContainerDescriptor desc = containerService.get(mboxRecordContainerUid);
		String mailboxAclUid = IMailboxAclUids.uidForMailbox(desc.owner);
		boolean canWriteOnMailbox = provider(bs).instance(IContainerManagement.class, mailboxAclUid)
				.canAccessVerbs(List.of(Verb.Write.name())).can();
		if (!canWriteOnMailbox) {
			return provider(bs).instance(IContainerManagement.class, mboxRecordContainerUid)
					.canAccessVerbs(List.of(Verb.Write.name())).can();
		}
		return canWriteOnMailbox;
	}

	private Vacation fromSet(Oof.Set set) {
		Vacation vacation = new Vacation();

		vacation.enabled = (set.oofState == null | set.oofState.equals(OofState.DISABLED)) ? false : true;
		if (set.startTime != null && !set.startTime.isBlank()) {
			vacation.start = parseDate(set.startTime);
		}
		if (set.endTime != null && !set.endTime.isBlank()) {
			vacation.end = parseDate(set.endTime);
		}
		if (set.oofMessages != null && !set.oofMessages.isEmpty()) {
			OofMessage selected = selectOofMessage(set.oofMessages);
			switch (selected.bodyType) {
			case SettingsResponse.Oof.BodyType.HTML -> vacation.textHtml = selected.replyMessage;
			default -> vacation.text = selected.replyMessage;
			}
			vacation.subject = "A vacation message";
		}
		return vacation;
	}

	private Date parseDate(String date) {
		try {
			return new Date(Instant.from(DateTimeFormatter.ISO_INSTANT.parse(date)).toEpochMilli());
		} catch (DateTimeException e) {
			logger.error("'Settings' 'Set' command : cannot parse '{}': {}", date, e.getMessage());
			return null;
		}
	}

	// Because only one message is allowed in BM, must select one among the three
	// availables
	private OofMessage selectOofMessage(List<OofMessage> messages) {
		Map<Audience, OofMessage> map = messages.stream().collect(Collectors.toMap(m -> m.audience, m -> m));
		if (map.containsKey(Audience.AppliesToExternalUnknown)
				&& !map.get(Audience.AppliesToExternalUnknown).replyMessage.isBlank()) {
			return map.get(Audience.AppliesToExternalUnknown);
		}
		if (map.containsKey(Audience.AppliesToExternalKnown)
				&& !map.get(Audience.AppliesToExternalKnown).replyMessage.isBlank()) {
			return map.get(Audience.AppliesToExternalKnown);
		}
		return map.get(Audience.AppliesToInternal);
	}

	@Override
	public long createEphemeralLongId(CollectionItem item) {
		byte[] cnt = new byte[16];
		ByteBuf b = Unpooled.wrappedBuffer(cnt).writerIndex(0);
		b.writeLong(item.collectionId.getFolderId()).writeLong(item.itemId);
		long reducedTo64Bits = Hashing.sipHash24().hashBytes(cnt).asLong() & 0x0FFFFFFFFFFFFFFFL;
		ephemeralLongIdMapping.put(reducedTo64Bits, cnt);
		return reducedTo64Bits;
	}

	@Override
	public Optional<CollectionItem> fromEphemeralLongId(long longId) {
		return Optional.ofNullable(ephemeralLongIdMapping.getIfPresent(longId)).map(colIdAndItemId -> {
			ByteBuf bb = Unpooled.wrappedBuffer(colIdAndItemId);
			long folderId = bb.readLong();
			long itemId = bb.readLong();
			return CollectionItem.of(CollectionId.of(Long.toString(folderId)), itemId);
		});
	}

}
