/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.central.reverse.proxy.health;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.LongAdder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.netty.handler.codec.http.HttpHeaderValues;
import io.vertx.core.AbstractVerticle;
import io.vertx.core.Promise;
import io.vertx.core.Verticle;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.json.JsonObject;
import net.bluemind.central.reverse.proxy.common.config.CrpConfig;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.IVerticlePriority;

public class HealthEndpointVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(HealthEndpointVerticle.class);

	private final LongAdder progressNotifications = new LongAdder();
	private final AtomicReference<String> kstreamState = new AtomicReference<>("UNKNOWN");
	private final AtomicLong modelLag = new AtomicLong(Integer.MAX_VALUE);
	private final AtomicLong streamLag = new AtomicLong(Integer.MAX_VALUE);
	private final Config config = CrpConfig.get("Health", getClass().getClassLoader());
	private final AtomicBoolean shutdownRequested = new AtomicBoolean(false);
	private long shutdownTimer = -1;

	@Override
	public void start(Promise<Void> startPromise) throws Exception {

		vertx.eventBus().consumer("dir.stream.state", (Message<String> stateMsg) -> kstreamState.set(stateMsg.body()));
		vertx.eventBus().consumer("dir.stream.lag", (Message<Long> lagMsg) -> streamLag.set(lagMsg.body()));
		vertx.eventBus().consumer("model.records.progress",
				(Message<String> progressMsg) -> progressNotifications.increment());
		vertx.eventBus().consumer("model.lag", (Message<Long> lagMsg) -> modelLag.set(lagMsg.body()));
		vertx.eventBus().consumer("shutdown.requested", s -> {
			Duration shutdownDelay = config.getDuration(CrpConfig.Health.GRACEFUL_SHUTDOWN_DELAY);
			shutdownRequested.set(true);
			if (shutdownTimer != -1) {
				logger.error("Forcibly shutting down as requested");
				System.exit(1);
			}
			logger.info("Graceful shutdown requested, stopping in {} seconds", shutdownDelay.toSeconds());
			shutdownTimer = vertx.setTimer(shutdownDelay.toMillis(), tId -> {
				System.exit(0);
			});
		});

		MessageProducer<Boolean> health = vertx.eventBus().publisher("crp.health");
		vertx.setPeriodic(Duration.ofSeconds(2).toMillis(), tid -> health.write(healthyCrp()));

		HttpServerOptions serverOptions = new HttpServerOptions().setTcpFastOpen(true) //
				.setTcpNoDelay(true) //
				.setAcceptBacklog(128) //
				.setReuseAddress(true) //
				.setReusePort(true);
		vertx.createHttpServer(serverOptions) //
				.requestHandler(request -> request.endHandler(v -> reportHealth(request)))//
				.invalidRequestHandler(invalidReq -> invalidReq.response().setStatusCode(400).end())//
				.exceptionHandler(t -> {
					/* HAProxy uses RST to stop the healthcheck */
					if (!t.getMessage().contains("Connection reset by peer")) {
						logger.error("HEALTH problem {}", t.getMessage(), t);
					}
				})//
				.listen(7070, ar -> {
					if (ar.failed()) {
						logger.error(ar.cause().getMessage(), ar.cause());
						startPromise.fail(ar.cause());
					} else {
						startPromise.complete();
						logger.info("Deployed {}", this);
					}
				});
	}

	private void reportHealth(HttpServerRequest request) {
		JsonObject status = new JsonObject();
		HttpServerResponse toRespond = request.response();
		toRespond.headers().add(HttpHeaders.CONTENT_TYPE, HttpHeaderValues.APPLICATION_JSON);
		status.put("stream", kstreamState.get());
		status.put("shuttingDown", shutdownRequested.get());
		status.put("modelNotifications", progressNotifications.sum());
		status.put("modelLag", modelLag.get());
		status.put("streamLag", streamLag.get());
		if (!shutdownRequested.get() && healthyCrp()) {
			status.put("status", "OK");
			toRespond.setStatusCode(200).end(status.encode());
		} else {
			status.put("status", "KO");
			toRespond.setStatusCode(503).end(status.encode());
		}
	}

	@Override
	public void stop(Promise<Void> stopPromise) throws Exception {
		logger.info("HEALTH Stopping....");
		stopPromise.complete();
	}

	private boolean healthyCrp() {
		long modelLagLimit = config.getLong(CrpConfig.Health.HEALTHY_MODEL_LAG);
		long streamLagLimit = config.getLong(CrpConfig.Health.HEALTHY_STREAM_LAG);

		return progressNotifications.sum() > 0 && "RUNNING".equals(kstreamState.get()) && modelLag.get() < modelLagLimit
				&& streamLag.get() < streamLagLimit;
	}

	public static class Factory implements IUniqueVerticleFactory, IVerticleFactory, IVerticlePriority {

		@Override
		public boolean isWorker() {
			return false;
		}

		@Override
		public Verticle newInstance() {
			return new HealthEndpointVerticle();
		}

		@Override
		public int getPriority() {
			return 5000;
		}

	}
}
