import { pki, jsbn } from "node-forge";

import { extensions } from "@bluemind/extensions";
import { enums, exceptions } from "@bluemind/smime.commons";

import getSMimeImplementation from "../smime/helpers/getSMimeImplementation";

import checkCertificate from "./checkCertificate";
import getCertificate from "./getCertificate";
import db from "./SMimePkiDB";

const { PKIStatus, SMIME_CERT_USAGE } = enums;
const { InvalidCertificateError, InvalidKeyError, MyKeyPairNotFoundError } = exceptions;

export { checkCertificate, getCertificate };

interface Cache {
    SIGNATURE_CERTIFICATE: pki.Certificate | undefined;
    SIGNATURE_PRIVATE_KEY: pki.rsa.PrivateKey | undefined;
    ENCRYPTION_CERTIFICATE: pki.Certificate | undefined;
    ENCRYPTION_PRIVATE_KEY: pki.rsa.PrivateKey | undefined;
}

const cache: Cache = {
    SIGNATURE_CERTIFICATE: undefined,
    SIGNATURE_PRIVATE_KEY: undefined,
    ENCRYPTION_CERTIFICATE: undefined,
    ENCRYPTION_PRIVATE_KEY: undefined
};

export async function getMyStatus(): Promise<enums.PKIStatus> {
    const implementation = getSMimeImplementation("getMyStatus");
    return implementation();
}
extensions.register("smime.implementations", "smime.service-worker", {
    getMyStatus: {
        fn: async function () {
            let status = PKIStatus.EMPTY;

            const { key: signatureKey, certificate: signatureCert } = await loadCachedKeyPair(
                SMIME_CERT_USAGE.SIGNATURE
            );

            if (signatureKey && signatureCert) {
                status |= PKIStatus.SIGNATURE_OK;
            }
            const { key: encryptionKey, certificate: encryptionCert } = await loadCachedKeyPair(
                SMIME_CERT_USAGE.ENCRYPTION
            );

            if (encryptionKey && encryptionCert) {
                status |= PKIStatus.ENCRYPTION_OK;
            }
            return status;
        },
        priority: 0
    }
});

export async function clear() {
    await db.clearMyCertsAndKeys();
    cache.SIGNATURE_CERTIFICATE = undefined;
    cache.SIGNATURE_PRIVATE_KEY = undefined;
    cache.ENCRYPTION_CERTIFICATE = undefined;
    cache.ENCRYPTION_PRIVATE_KEY = undefined;
    await db.clearRevocations();
}

export async function getMySignatureKeyPair(
    validity?: Date
): Promise<{ key: pki.rsa.PrivateKey; certificate: pki.Certificate }> {
    const implementation = getSMimeImplementation("getMySignatureKeyPair");
    return implementation(validity);
}
extensions.register("smime.implementations", "smime.service-worker", {
    getMySignatureKeyPair: {
        fn: async function (validity?: Date) {
            return getValidKeyPair(SMIME_CERT_USAGE.SIGNATURE, validity);
        }
    },
    priority: 0
});

export async function getMyEncryptionCertificate(validity?: Date): Promise<pki.Certificate | string> {
    const implementation = getSMimeImplementation("getMyEncryptionCertificate");
    return implementation(validity);
}
extensions.register("smime.implementations", "smime.service-worker", {
    getMyEncryptionCertificate: {
        fn: async function (validity?: Date) {
            const { certificate } = await getValidKeyPair(SMIME_CERT_USAGE.ENCRYPTION, validity);
            return certificate;
        }
    },
    priority: 0
});

export async function getMyDecryptionKeyPair(
    validity?: Date
): Promise<{ key: pki.rsa.PrivateKey; certificate: pki.Certificate }> {
    const implementation = getSMimeImplementation("getMyDecryptionKeyPair");
    return implementation(validity);
}
extensions.register("smime.implementations", "smime.service-worker", {
    getMyDecryptionKeyPair: {
        fn: async function (validity?: Date) {
            return getValidKeyPair(SMIME_CERT_USAGE.ENCRYPTION, validity);
        }
    },
    priority: 0
});

export async function getAllDecryptionKeyPairs(
    validity: Date
): Promise<{ key: pki.rsa.PrivateKey; certificate: pki.Certificate }[]> {
    const implementation = getSMimeImplementation("getAllDecryptionKeyPairs");
    return implementation(validity);
}
extensions.register("smime.implementations", "smime.service-worker", {
    getAllDecryptionKeyPairs: {
        fn: async function (validity: Date) {
            return getKeyPairs(SMIME_CERT_USAGE.ENCRYPTION, validity);
        }
    },
    priority: 0
});

async function getValidKeyPair(
    usage: enums.SMIME_CERT_USAGE,
    validityDate?: Date
): Promise<{ key: pki.rsa.PrivateKey; certificate: pki.Certificate }> {
    const date = validityDate ?? new Date();
    const cached = await loadCachedKeyPair(usage);
    if (cached.certificate && cached.key && isValid(cached.certificate, date)) {
        return { key: cached.key, certificate: cached.certificate };
    }

    const pairs = await getKeyPairs(usage, date);
    if (pairs.length === 0) {
        throw new MyKeyPairNotFoundError(date);
    }

    return pairs[0];
}

function isValid(certificate: pki.Certificate, validityDate: Date): boolean {
    const { notBefore, notAfter } = certificate.validity;
    return notBefore <= validityDate && validityDate <= notAfter;
}

export async function setMyPrivateKeys(blob: Blob) {
    await db.setPrivateKeys(blob);
}
export async function setMyCertificates(blob: Blob) {
    await db.setCertificates(blob);
}

async function getPrivateKeys(): Promise<pki.PEM[]> {
    const blob = (await db.getPrivateKeys()) as Blob;
    const text = await blob?.text?.();
    try {
        const keys = text ? JSON.parse(text) : [];
        return keys;
    } catch (error) {
        throw new InvalidKeyError(error);
    }
}

async function getCertificates(): Promise<pki.PEM[]> {
    const blob = (await db.getCertificates()) as Blob;
    const text = await blob?.text?.();
    try {
        const certificates = text ? JSON.parse(text) : [];
        return certificates;
    } catch (error) {
        throw new InvalidCertificateError(error);
    }
}

async function loadCachedKeyPair(keyUsage: enums.SMIME_CERT_USAGE): Promise<{
    key?: pki.rsa.PrivateKey;
    certificate?: pki.Certificate;
}> {
    const cacheUsage = keyUsage || SMIME_CERT_USAGE.SIGNATURE;
    const PRIVATE_KEY: "SIGNATURE_PRIVATE_KEY" | "ENCRYPTION_PRIVATE_KEY" = `${cacheUsage}_PRIVATE_KEY`;
    const CERTIFICATE: "SIGNATURE_CERTIFICATE" | "ENCRYPTION_CERTIFICATE" = `${cacheUsage}_CERTIFICATE`;
    if (!cache[CERTIFICATE] || !cache[PRIVATE_KEY]) {
        const today = new Date();
        const [keyPair] = await getKeyPairs(keyUsage, today);
        cache[PRIVATE_KEY] = keyPair?.key;
        cache[CERTIFICATE] = keyPair?.certificate;
    }
    return { certificate: cache[CERTIFICATE], key: cache[PRIVATE_KEY] };
}

async function getKeyPairs(
    keyUsage: enums.SMIME_CERT_USAGE,
    date: Date
): Promise<{ key: pki.rsa.PrivateKey; certificate: pki.Certificate }[]> {
    const pairs = [];
    const certificates = await getCertificates();

    for (const certPem of certificates) {
        const certificate = pki.certificateFromPem(certPem);
        try {
            await checkCertificate(certificate, { smimeUsage: keyUsage, date });
        } catch {
            continue;
        }
        const key = await getAssociatedKey(certificate);
        if (key && certificate) {
            pairs.push({ key, certificate });
        }
    }
    return pairs;
}

async function getAssociatedKey(certificate: pki.Certificate): Promise<pki.rsa.PrivateKey | undefined> {
    type Key = {
        n: jsbn.BigInteger;
        e: jsbn.BigInteger;
    };
    const certPublicKey = certificate.publicKey as Key;
    const keys = await getPrivateKeys();
    for (const keyPem of keys) {
        const privateKey = pki.privateKeyFromPem(keyPem);
        const isSameKey = privateKey.n.equals(certPublicKey.n) && privateKey.e.equals(certPublicKey.e);
        if (isSameKey) {
            return privateKey;
        }
    }
}
