import parse from "emailjs-mime-parser";
import { md, pkcs7, pki, asn1, util, Encoding } from "node-forge";

import type { MessageBody } from "@bluemind/backend.mail.api";
import { exceptions } from "@bluemind/smime.commons";
const { InvalidMessageIntegrityError, InvalidPkcs7EnvelopeError, InvalidSignatureError, UnsupportedAlgorithmError } =
    exceptions;

export function getSigningTime(envelope: pkcs7.Captured<pkcs7.PkcsSignedData>) {
    const parent = getParent(envelope.rawCapture.authenticatedAttributesAsn1, pki.oids["signingTime"]);
    if (!parent) {
        throw new InvalidPkcs7EnvelopeError();
    }
    return asn1.utcTimeToDate(<string>(<asn1.Asn1>(<asn1.Asn1>parent.value[1]).value[0]).value);
}

export function checkSignatureValidity(envelope: pkcs7.Captured<pkcs7.PkcsSignedData>, certificate: pki.Certificate) {
    function verifyAttributes(algorithm: string): boolean {
        const authAttributesDigest = digestAttributes(envelope.rawCapture.authenticatedAttributesAsn1, algorithm);
        const pubKey = <pki.rsa.PublicKey>certificate.publicKey;
        return pubKey.verify(authAttributesDigest, envelope.rawCapture.signature, "RSASSA-PKCS1-V1_5");
    }
    // The signature should be verified with the certificate.siginfo.algorithmOid but sometimes
    // there is a mismatch between the certificate's algorithm identifier and the actual signature algorithm
    const preferedAlgorithm = certificate.siginfo.algorithmOid;
    const algorithms = [preferedAlgorithm, ...Object.keys(digestAlgorithms)];
    const isVerified = algorithms.some(algorithmOid => verifyAttributes(algorithmOid));
    if (!isVerified) {
        throw new InvalidSignatureError();
    }
}

export function checkMessageIntegrity(
    envelope: pkcs7.Captured<pkcs7.PkcsSignedData>,
    contentToDigest: string,
    bodyStructure: MessageBody.Part
) {
    const encoding: Encoding = hasUtf8Charset(bodyStructure) ? "utf8" : "raw";
    const algo = getHashFunction(asn1.derToOid(envelope.rawCapture.digestAlgorithm));
    const normalized = contentToDigest.replace(/(?<!\r)\n/g, "\r\n");
    const computedDigest = algo.create().update(normalized, encoding).digest().bytes();
    const messageDigest = getMessageDigest(envelope.rawCapture.authenticatedAttributesAsn1);
    if (util.bytesToHex(messageDigest) !== util.bytesToHex(computedDigest)) {
        throw new InvalidMessageIntegrityError();
    }
}

const digestAlgorithms = {
    [pki.oids.md5WithRSAEncryption]: md.md5,
    [pki.oids.sha1WithRSAEncryption]: md.sha1,
    [pki.oids.sha256WithRSAEncryption]: md.sha256,
    [pki.oids.sha384WithRSAEncryption]: md.sha384,
    [pki.oids.sha512WithRSAEncryption]: md.sha512
};

function digestAttributes(attrs: pki.Attribute, algorithm: string): string {
    // per RFC 2315, attributes are to be digested using a SET container
    // not the above [0] IMPLICIT container
    const attrsAsn1 = asn1.create(asn1.Class.UNIVERSAL, asn1.Type.SET, true, []);
    attrsAsn1.value = attrs.value;

    // DER-serialize and digest SET OF attributes only
    const bytes = asn1.toDer(attrsAsn1).getBytes();

    const algo = getDigestAttrAlgo(algorithm);
    return algo.update(bytes).digest().getBytes();
}

function getDigestAttrAlgo(algorithmOid: string) {
    if (Object.keys(digestAlgorithms).includes(algorithmOid)) {
        return digestAlgorithms[algorithmOid].create();
    } else {
        throw new UnsupportedAlgorithmError(pki.oids[algorithmOid]);
    }
}

function getHashFunction(algorithmOid: string) {
    switch (algorithmOid) {
        case pki.oids.md5:
            return md.md5;
        case pki.oids.sha1:
            return md.sha1;
        case pki.oids.sha256:
            return md.sha256;
        case pki.oids.sha384:
            return md.sha384;
        case pki.oids.sha512:
            return md.sha512;
        default:
            throw new UnsupportedAlgorithmError(pki.oids[algorithmOid]);
    }
}

function getMessageDigest(attribute: asn1.Asn1): string {
    const parent = getParent(attribute, pki.oids["messageDigest"]);
    if (!parent) {
        throw new InvalidPkcs7EnvelopeError();
    }
    return <string>(<asn1.Asn1>(<asn1.Asn1>parent.value[1]).value[0]).value;
}

function getParent(attribute: asn1.Asn1, searchedOid: string): asn1.Asn1 | undefined {
    if (isOid(attribute, searchedOid)) {
        return attribute; // no parent so return myself
    } else if (Array.isArray(attribute.value)) {
        let index = 0;
        do {
            if (getParent(attribute.value[index], searchedOid)) {
                return attribute.value[index];
            }
            index++;
        } while (index < attribute.value.length);
    }
}

function isOid(attr: asn1.Asn1, oid: string): boolean {
    return attr.type === asn1.Type.OID && asn1.derToOid(new util.ByteStringBuffer(<string>attr.value)) === oid;
}

export function getContentFromAttachedSignature(eml: string): string {
    const parsed = parse(eml);
    const asn1Data = asn1.fromDer(new util.ByteStringBuffer(parsed.content));
    const msg = pkcs7.messageFromAsn1(asn1Data);

    function getValue(obj: asn1.Asn1): string {
        return obj.composed ? (obj.value as asn1.Asn1[]).map(getValue).join("") : (obj.value as string);
    }

    return getValue(msg.rawCapture.content);
}

function hasUtf8Charset(part: MessageBody.Part): boolean {
    if (part.charset === "utf-8") {
        return true;
    }
    if (part.children && part.children.length > 0) {
        return part.children.some((child: MessageBody.Part) => hasUtf8Charset(child));
    }
    return false;
}
