import { pkcs7, pki } from "node-forge";

import { MessageBody } from "@bluemind/backend.mail.api";
import { enums } from "@bluemind/smime.commons";

import { checkCertificate } from "../../pki/";

import { checkMessageIntegrity, checkSignatureValidity, getSigningTime } from "./verify";

const { SMIME_CERT_USAGE } = enums;

export default async function verify(
    envelope: pkcs7.Captured<pkcs7.PkcsSignedData>,
    toDigest: string,
    body: MessageBody
) {
    const sender = body.recipients?.find(recipient => recipient.kind === "Originator")?.address as string;
    const certificate = await findValidCertificate(envelope, sender);
    checkSignatureValidity(envelope, certificate);
    checkMessageIntegrity(envelope, toDigest, body.structure!);
}

async function findValidCertificate(
    envelope: pkcs7.Captured<pkcs7.PkcsSignedData>,
    expectedAddress: string
): Promise<pki.Certificate> {
    const date = getSigningTime(envelope);

    let certificate, error;
    for (certificate of envelope.certificates) {
        try {
            await checkCertificate(certificate, { date, expectedAddress, smimeUsage: SMIME_CERT_USAGE.SIGNATURE });
            return certificate;
        } catch (err) {
            error = err;
        }
    }
    throw error;
}
