import { Flag } from "@bluemind/email";
import i18n from "@bluemind/i18n";
import { inject } from "@bluemind/inject";
import { folderUtils, messageUtils } from "@bluemind/mail";
import { retrieveTaskResult } from "@bluemind/task";

import { ADD_FLAG } from "~/actions";
import { REMOVE_MESSAGES, SET_MESSAGES_STATUS } from "~/mutations";

import apiMessages from "../../api/apiMessages";

import { scheduleAction, Actions } from "./draftActionsScheduler";

const { MessageStatus, MessageAdaptor, MessageHeader, MessageCreationModes, isScheduled } = messageUtils;

export default async function (context, args) {
    return scheduleAction(() => send(context, args), Actions.SEND);
}

async function send(context, { draft, myMailboxKey, outbox, myDraftsFolder }) {
    let message = context.state[draft.key];
    const isDraft = message.folderRef.uid === myDraftsFolder.remoteRef.uid;

    context.commit(SET_MESSAGES_STATUS, [{ key: message.key, status: MessageStatus.SENDING }]);

    const messageInOutboxId = await moveToOutbox(
        message,
        myMailboxKey,
        outbox.remoteRef.internalId,
        outbox.remoteRef.uid,
        myDraftsFolder.remoteRef.internalId
    );

    const flushPromise = flush(); // flush means send mail + move to sentbox
    const taskResult = await flushPromise;
    context.commit(SET_MESSAGES_STATUS, [{ key: message.key, status: MessageStatus.IDLE }]);
    if (isDraft) updateDraftStatus(context, message);

    return getMessageSent(message, taskResult, messageInOutboxId, outbox);
}

function updateDraftStatus(context, draft) {
    context.commit(REMOVE_MESSAGES, { messages: [draft] });
    manageFlagOnPreviousMessage(context, draft);
    apiMessages.removeParts(draft);
}

function getMessageSent(message, taskResult, messageInOutboxId, outbox) {
    return isScheduled(message)
        ? getScheduledMessage(messageInOutboxId, outbox)
        : getSentMessage(taskResult, messageInOutboxId, outbox);
}

async function getSentMessage(taskResult, messageInOutboxId, outbox) {
    const flushResult = taskResult.result.find(
        r =>
            r.sourceFolderUid.toUpperCase() === outbox.remoteRef.uid.toUpperCase() &&
            r.sourceInternalId === messageInOutboxId
    );
    const mailItem = await inject("MailboxItemsPersistence", flushResult.destinationFolderUid).getCompleteById(
        flushResult.destinationInternalId
    );
    return MessageAdaptor.fromMailboxItem(mailItem, {
        key: flushResult.destinationFolderUid,
        uid: flushResult.destinationFolderUid
    });
}

async function getScheduledMessage(messageInOutboxId, outbox) {
    const mailItem = await inject("MailboxItemsPersistence", outbox.remoteRef.uid).getCompleteById(messageInOutboxId);
    return MessageAdaptor.fromMailboxItem(mailItem, {
        key: outbox.remoteRef.uid,
        uid: outbox.remoteRef.uid
    });
}

function flush() {
    return inject("OutboxPersistence")
        .flush()
        .then(taskRef => {
            // wait for the flush of the outbox to be finished
            const taskService = inject("TaskService", taskRef.id);
            return retrieveTaskResult(taskService);
        });
}

async function moveToOutbox(message, myMailboxKey, outboxId, outboxUid, myDraftsFolderId) {
    if (message.folderRef.uid === outboxUid) {
        return message.remoteRef.internalId;
    }
    validateDraft(message);
    const moveResult = await inject("MailboxFoldersPersistence", myMailboxKey).importItems(outboxId, {
        mailboxFolderId: myDraftsFolderId,
        ids: [{ id: message.remoteRef.internalId }],
        deleteFromSource: true
    });
    if (!moveResult || moveResult.status !== "SUCCESS") {
        throw "Unable to move draft to Outbox.";
    }
    return moveResult.doneIds[0].destination;
}

function manageFlagOnPreviousMessage({ dispatch, state }, draft) {
    const hasDraftInfoHeader = draft.headers.find(header => header.name === MessageHeader.X_BM_DRAFT_INFO);
    if (hasDraftInfoHeader) {
        const draftInfoHeader = JSON.parse(hasDraftInfoHeader.values[0]);
        const mailboxItemFlag = draftInfoHeader.type === MessageCreationModes.FORWARD ? Flag.FORWARDED : Flag.ANSWERED;

        const messageInternalId = draftInfoHeader.messageInternalId;
        const folderUid = draftInfoHeader.folderUid;
        const folderKey = folderUtils.generateKey(folderUid);
        const messageKey = messageUtils.messageKey(messageInternalId, folderKey);
        const message = state[messageKey];
        if (message) {
            dispatch(ADD_FLAG, {
                messages: [message],
                flag: mailboxItemFlag
            });
        } else {
            inject("MailboxItemsPersistence", folderUid).addFlag({ itemsId: [messageInternalId], mailboxItemFlag });
        }
    }
}

function validateDraft(draft) {
    const recipients = draft.to.concat(draft.cc).concat(draft.bcc).concat(draft.replyTo);
    if (!recipients.length > 0) {
        throw i18n.global.t("mail.error.email.address.invalid");
    }
}
