import { markRaw } from "vue";

import { AlertTypes } from "@bluemind/alert.store";
import { inject } from "@bluemind/inject";
import { signatureUtils, mailTipUtils } from "@bluemind/mail";

import {
    DEBOUNCED_SET_MESSAGE_CONTENT,
    LOAD_MAX_MESSAGE_SIZE,
    REMOVE_ATTACHMENT,
    SET_DRAFT_CONTENT,
    SET_MESSAGE_CONTENT,
    TOGGLE_SIGNATURE
} from "~/actions";
import {
    DISCLAIMER,
    GET_DRAFT_CONTENT,
    IS_REPLY_TO_SHOWN,
    IS_SENDER_SHOWN,
    MAILTIPS_ALERTS,
    MAILTIPS_OOF,
    MAILTIPS_OVER_QUOTA,
    SIGNATURE
} from "~/getters";
import {
    ADD_FILE,
    RESET_COMPOSER,
    RESET_FILES,
    SET_DRAFT_EDITOR_CONTENT,
    SET_FILE_ADDRESS,
    SET_FILE_HEADERS,
    SET_FILE_PROGRESS,
    SET_FILE_STATUS,
    SET_MAIL_TIPS,
    SET_MAX_MESSAGE_SIZE,
    SET_PERSONAL_SIGNATURE,
    SHOW_REPLY_TO,
    SHOW_SENDER,
    SIGNATURE_TOGGLED,
    SHOW_SEND_LATER_MODAL,
    SET_SHOW_FORMATTING_TOOLBAR,
    SET_INPUT_VALUE_FIELD_TO,
    SET_INPUT_VALUE_FIELD_CC,
    SET_INPUT_VALUE_FIELD_BCC,
    SET_INPUT_VALUE_FIELD_REPLYTO,
    SET_INVALID_RECIPIENT_ALERT,
    SET_SENDING
} from "~/mutations";

import MailtipAlert from "../components/MailAlerts/MailtipAlert.vue";
import MailtipOOF from "../components/MailAlerts/MailtipOOF.vue";
import MailtipOverQuota from "../components/MailAlerts/MailtipOverQuota.vue";

import templateChooser from "./templateChooser";

const {
    removeCorporateSignatureContent,
    isCorporateSignature,
    isDisclaimer,
    wrapCorporateSignature,
    wrapDisclaimer,
    wrapPersonalSignature
} = signatureUtils;

const { MailTipTypes } = mailTipUtils;

const MAILTIP_CONFIGS = {
    OVER_QUOTA: {
        tipType: MailTipTypes.OVER_QUOTA,
        maxDisplayed: 1,
        component: markRaw(MailtipOverQuota),
        icon: "inbox-full",
        recipientStateKey: "overQuotaRecipients",
        filterValue: true
    },
    OUT_OF_OFFICE: {
        tipType: MailTipTypes.OUT_OF_OFFICE,
        maxDisplayed: 3,
        component: markRaw(MailtipOOF),
        icon: "magnetic-tape",
        recipientStateKey: "outOfOfficeRecipients",
        filterValue: false
    }
};

export default {
    modules: {
        templateChooser
    },
    state: {
        personalSignature: { toggleStatus: false },
        editorContent: "",
        inlineImagesSaved: [],
        maxMessageSize: 0, // FIXME: it's a cross-composer data, it must be moved in another store
        isSenderShown: false,
        isReplyToShown: false,
        maxMessageSizeExceeded: false,
        showFormattingToolbar: false,
        sendLaterModal: { visible: false },
        synced: ["showFormattingToolbar"],
        mailTips: [],
        uploadingFiles: {},
        inputValueTo: "",
        inputValueCc: "",
        inputValueBcc: "",
        inputValueReplyto: "",
        invalidRecipientAlert: false,
        outOfOfficeRecipients: [],
        overQuotaRecipients: [],
        sending: false
    },

    mutations: {
        [RESET_COMPOSER]: (state, { isSenderShown = false }) => {
            state.personalSignature = { toggleStatus: false };
            state.editorContent = "";
            state.inlineImagesSaved = [];
            state.isSenderShown = isSenderShown;
            state.isReplyToShown = false;
            state.maxMessageSizeExceeded = false;
            state.invalidRecipientAlert = false;
            state.inputValueTo = "";
            state.inputValueCc = "";
            state.inputValueBcc = "";
            state.inputValueReplyto = "";
        },
        [SET_PERSONAL_SIGNATURE]: (state, signature) => {
            state.personalSignature = { toggleStatus: state.personalSignature.toggleStatus, ...signature };
        },
        [SET_DRAFT_EDITOR_CONTENT]: (state, content) => {
            state.editorContent = content;
        },
        [SET_MAX_MESSAGE_SIZE](state, size) {
            state.maxMessageSize = size;
        },
        [SHOW_REPLY_TO]: (state, value) => {
            state.isReplyToShown = value;
        },
        [SHOW_SENDER]: (state, value) => {
            state.isSenderShown = value;
        },

        [SET_MAIL_TIPS]: (state, mailTips) => {
            state.outOfOfficeRecipients = [];
            state.overQuotaRecipients = [];
            state.mailTips = mailTips;
        },
        [SET_SHOW_FORMATTING_TOOLBAR]: (state, value) => {
            state.showFormattingToolbar = value;
        },
        SET_SAVE_ERROR: (state, error) => {
            state.maxMessageSizeExceeded = error?.data?.errorCode === "ENTITY_TOO_LARGE";
        },
        [RESET_FILES]: state => {
            state.uploadingFiles = {};
        },
        [ADD_FILE]: (state, { file }) => {
            state.uploadingFiles[file.key] = file;
        },
        [SET_FILE_PROGRESS]: (state, { key, progress }) => {
            updateUploadingFiles(state, key, { progress });
        },
        [SET_FILE_STATUS]: (state, { key, status }) => {
            updateUploadingFiles(state, key, { status });
        },
        [SET_FILE_HEADERS]: (state, { key, headers }) => {
            updateUploadingFiles(state, key, { headers });
        },
        [SET_FILE_ADDRESS]: (state, { key, address }) => {
            updateUploadingFiles(state, key, { address });
        },

        // Listeners
        [REMOVE_ATTACHMENT]: (state, { address }) => {
            const index = Object.values(state.uploadingFiles).findIndex(file => address === file.address);
            if (index > -1) {
                const key = Object.keys(state.uploadingFiles)[index];
                delete state.uploadingFiles[key];
            }
        },
        [SIGNATURE_TOGGLED]: (state, payload) => {
            state.personalSignature.toggleStatus = payload;
        },
        [SHOW_SEND_LATER_MODAL]: (state, isVisible) => {
            state.sendLaterModal.visible = isVisible;
        },
        [SET_INPUT_VALUE_FIELD_TO]: (state, payload) => {
            state.inputValueTo = payload;
        },
        [SET_INPUT_VALUE_FIELD_CC]: (state, payload) => {
            state.inputValueCc = payload;
        },
        [SET_INPUT_VALUE_FIELD_BCC]: (state, payload) => {
            state.inputValueBcc = payload;
        },
        [SET_INPUT_VALUE_FIELD_REPLYTO]: (state, payload) => {
            state.inputValueReplyto = payload;
        },
        [SET_INVALID_RECIPIENT_ALERT]: (state, payload) => {
            state.invalidRecipientAlert = payload;
        },
        [SET_SENDING]: (state, payload) => {
            state.sending = payload;
        }
    },

    actions: {
        async [LOAD_MAX_MESSAGE_SIZE]({ commit }, userId) {
            const { messageMaxSize } = await inject("MailboxesPersistence").getMailboxConfig(userId);
            // take into account the email base64 encoding : 33% more space
            commit(SET_MAX_MESSAGE_SIZE, messageMaxSize / 1.33);
        },
        [SET_DRAFT_CONTENT]: setDraftContent,
        [TOGGLE_SIGNATURE]({ state, commit }) {
            commit(SIGNATURE_TOGGLED, !state.personalSignature.toggleStatus);
        }
    },

    getters: {
        [GET_DRAFT_CONTENT]: ({ editorContent }, getters) => {
            const signature = getters[SIGNATURE];
            const corporateSignature = signature?.uid ? signature : null;
            const disclaimer = getters[DISCLAIMER];
            return removeCorporateSignatureContent(editorContent, { corporateSignature, disclaimer });
        },
        [IS_REPLY_TO_SHOWN]: state => state.isReplyToShown,
        [IS_SENDER_SHOWN]: state => state.isSenderShown,

        [SIGNATURE]: (state, getters) => {
            if (!state.mailTips.length && state.personalSignature) {
                return getters["__fallbackSignature"];
            }

            const matchingTips = state.mailTips[0]?.matchingTips || [];
            const corporateSignature = matchingTips.find(isCorporateSignature);
            if (!corporateSignature) {
                return getters["__fallbackSignature"];
            }

            const jsonCorporateSignature = JSON.parse(corporateSignature.value);
            return {
                ...jsonCorporateSignature,
                html: wrapCorporateSignature(jsonCorporateSignature.html),
                isCorporate: true
            };
        },
        __fallbackSignature: state => {
            if (state.personalSignature.toggleStatus) {
                return {
                    id: state.personalSignature.id,
                    html: wrapPersonalSignature({
                        html: state.personalSignature?.html,
                        id: state.personalSignature?.id
                    })
                };
            }
            return null;
        },
        [DISCLAIMER]: state => {
            if (!state.mailTips.length) {
                return null;
            }

            const matchingTips = state.mailTips[0].matchingTips;
            const disclaimer = matchingTips.find(isDisclaimer)?.value;

            if (!disclaimer) {
                return null;
            }

            return wrapDisclaimer(JSON.parse(disclaimer).html);
        },
        [MAILTIPS_ALERTS]: state => {
            if (!state.mailTips.length) {
                return null;
            }

            const matchingTips = state.mailTips[0].matchingTips;
            const alerts = matchingTips.flatMap(matchingTip =>
                isAlert(matchingTip) ? wrapAlert(JSON.parse(matchingTip.value)) : []
            );

            return alerts.length ? alerts : null;
        },
        [MAILTIPS_OVER_QUOTA]: state => mailTipHandler(state, MAILTIP_CONFIGS.OVER_QUOTA),
        [MAILTIPS_OOF]: state => mailTipHandler(state, MAILTIP_CONFIGS.OUT_OF_OFFICE)
    }
};

function isAlert(matchingTip) {
    return matchingTip.mailtipType === MailTipTypes.ALERT;
}

function wrapAlert(alertTipValue) {
    const level = alertTipValue.level.toUpperCase();
    const alertType = AlertTypes[level] ?? AlertTypes.WARNING;
    return {
        alert: {
            uid: `MAILTIP_ALERT_${alertTipValue.id}`,
            type: alertType,
            renderer: { component: markRaw(MailtipAlert), props: { html: alertTipValue.text } }
        },
        options: {
            icon: alertTipValue.icon,
            countDown: alertTipValue.countDown,
            dismissible: alertTipValue.dismissible === undefined ? true : alertTipValue.dismissible
        }
    };
}

function mailTipHandler(state, config) {
    if (!state.mailTips.length) {
        return null;
    }
    const { tipType, maxDisplayed, component, icon, recipientStateKey, filterValue } = config;

    const tips = filterMailTips(state.mailTips, tipType, filterValue);
    state[recipientStateKey] = tips.map(tip => tip.recipient.email);

    if (tips.length > maxDisplayed) {
        return [wrapMailTips(tips, tipType, component, icon)];
    } else if (tips.length) {
        return tips.map(tip => wrapMailTip(tip, tipType, component, icon));
    }

    return null;
}

function filterMailTips(mailTips, type, filterValue) {
    const seenEmails = new Set();
    return mailTips
        .flatMap(({ matchingTips, forRecipient }) =>
            matchingTips
                .filter(tip => tip.mailtipType === type && (!filterValue || tip.value === "true"))
                .map(tip => ({ recipient: forRecipient, value: tip.value }))
        )
        .filter(({ recipient }) => {
            if (seenEmails.has(recipient.email)) {
                return false;
            }
            seenEmails.add(recipient.email);
            return true;
        });
}

function wrapMailTip(tip, type, component, icon) {
    const uid = `MAILTIP_${type}_${tip.recipient.email}`;
    return {
        alert: {
            uid,
            type: AlertTypes.INFO,
            renderer: {
                component,
                props: { alertUid: uid, tips: [tip], multi: false }
            }
        },
        options: {
            icon,
            dismissible: true
        }
    };
}

function wrapMailTips(tips, type, component, icon) {
    const uid = `MAILTIP_${type}_MULTIPLE_${tips.length}`;
    return {
        alert: {
            uid,
            type: AlertTypes.INFO,
            renderer: {
                component,
                props: { alertUid: uid, tips: [...tips], multi: true }
            }
        },
        options: {
            icon,
            dismissible: true
        }
    };
}

function updateUploadingFiles(state, key, update) {
    if (!state.uploadingFiles[key]) {
        state.uploadingFiles[key] = {};
    }
    const file = state.uploadingFiles[key];
    Object.keys(update).forEach(keyEntry => {
        const value = update[keyEntry];
        if (!(keyEntry in file)) {
            file[keyEntry] = value;
        }
        file[keyEntry] = value;
    });
}

function setDraftContent({ commit, getters, dispatch }, { draft, html, debounce }) {
    commit(SET_DRAFT_EDITOR_CONTENT, html);
    const content = getters[GET_DRAFT_CONTENT];
    const action = debounce === false ? SET_MESSAGE_CONTENT : DEBOUNCED_SET_MESSAGE_CONTENT;
    return dispatch(action, { message: draft, content });
}
