import { messageUtils } from "@bluemind/mail";

const { MessageForwardAttributeSeparator, MessageReplyAttributeSeparator, MessageQuoteClasses, MessageQuoteOutlookId } =
    messageUtils;

const NOT_FOUND = "NOT_FOUND";

export default {
    NOT_FOUND,

    findQuoteNodes(message, htmlDoc) {
        return (
            findLastTopLevelBlockquote(htmlDoc) ||
            findFromNodeAndNextSiblings(htmlDoc, message) ||
            findReplyOrForwardContentNodesNotInsideBlockquote(htmlDoc)
        );
    },

    /**
     * @returns the nodes in 'html' containing 'quote' and wrapping elements ('blockquote' tag, 'has wrote' header...).
     */
    findQuoteNodesUsingTextComparison(html, quote) {
        const rootDocument = new DOMParser().parseFromString(html, "text/html");
        const quoteElement = new DOMParser().parseFromString(quote, "text/html").body;
        const nodeWithQuoteInRootNode = findNodeContainingTextNodes(rootDocument, quoteElement);
        if (nodeWithQuoteInRootNode) {
            const nodeIterator = rootDocument.createNodeIterator(rootDocument.body, NodeFilter.SHOW_ALL, node =>
                node.isEqualNode(nodeWithQuoteInRootNode) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT
            );
            const quoteNode = nodeIterator.nextNode(); // assumes the first one is the only one needed
            const quoteNodes = addQuoteHeaderAndWrappingElements(quoteNode);
            return quoteNodes;
        }
    },

    collapseQuotes(rootNode, quoteNodes) {
        if (quoteNodes) {
            const details = document.createElement("details");
            const summary = document.createElement("summary");
            details.appendChild(summary);
            const nodeIterator = rootNode.createNodeIterator(rootNode, NodeFilter.SHOW_ALL, node =>
                quoteNodes.some(qn => node.isSameNode(qn)) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT
            );
            let node;
            let detailsInserted = false;
            const nodes = [];
            while ((node = nodeIterator.nextNode())) {
                if (detailsInserted) {
                    nodes.push(node.parentNode.removeChild(node));
                } else {
                    nodes.push(node.parentNode.replaceChild(details, node));
                    detailsInserted = true;
                }
            }
            nodes.forEach(node => details.appendChild(node));
            return rootNode;
        }
    }
};

const SEP_IDS = [MessageReplyAttributeSeparator, MessageForwardAttributeSeparator, MessageQuoteOutlookId];
const SEP_CLASSES = MessageQuoteClasses;

function findReplyOrForwardContentNodesNotInsideBlockquote(htmlDoc) {
    // xpath example: //*[(@id="data-bm-reply-separator" or @id="data-bm-forward-separator" or contains(class,"data-bm-reply-separator") or contains(class,"data-bm-forward-separator")) and not(ancestor::blockquote)]
    const sepIdXpath = SEP_IDS.length ? SEP_IDS.map(sid => '@id="' + sid + '"').join(" or ") : "";
    const sepClassXpath = SEP_CLASSES.length
        ? " or " + SEP_CLASSES.map(sc => 'contains(@class,"' + sc + '")').join(" or ")
        : "";
    const xpath = `//*[(${sepIdXpath}${sepClassXpath}) and not(ancestor::blockquote)]`;
    const xpathResult = htmlDoc?.evaluate(xpath, htmlDoc.body, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
    let res = xpathResult?.iterateNext();
    const nodes = res ? [getHighestAncestorWithSameContent(res)] : undefined;
    while (res) {
        res = xpathResult?.iterateNext();
        if (res) {
            nodes.push(getHighestAncestorWithSameContent(res));
        }
    }
    return nodes;
}

/** Find 'from' contact node (like "Georges Abitbol <g.abitol@lca.net>") and its following siblings. */
function findFromNodeAndNextSiblings(htmlDoc, message) {
    // find "lines" containing "To" recipient
    const bodyText = htmlDoc.body.textContent;
    const toRegex =
        ".*?(?:" +
        message.to.reduce((all, current) => {
            const allStr = all ? `${all}|` : "";
            const currentStr = current.dn ? `${current.dn}|${current.address}` : current.address;
            return `${allStr}${currentStr}`;
        }, "") +
        ").*";
    const matches = Array.from(new Set(bodyText.match(new RegExp(toRegex))));

    if (matches) {
        for (let i = matches.length - 1; i >= 0; i--) {
            const matchingLine = matches[i]?.trim();
            const previousText = bodyText.split(matchingLine)[0];

            const nodeWalker = (node, fn) => {
                fn(node);
                node?.childNodes.forEach(child => nodeWalker(child, fn));
            };

            let fromNode;
            nodeWalker(htmlDoc.body, node => {
                if (
                    node?.textContent?.includes(matchingLine) &&
                    (!previousText || !node?.textContent?.includes(previousText))
                ) {
                    fromNode = node;
                }
            });
            if (fromNode) {
                const node = getHighestAncestorWithSameContent(fromNode);
                const nodes = [node];
                const isOnlyChild = node.parentNode.children.length === 1;
                let sibling = isOnlyChild ? node.parentElement | {} : node;
                while ((sibling = sibling.nextElementSibling)) {
                    nodes.push(sibling);
                }
                return nodes;
            }
        }
    }
}

function getHighestAncestorWithSameContent(node) {
    let currentNode = node;
    while (
        currentNode?.parentElement &&
        currentNode?.parentElement.tagName !== "BODY" &&
        currentNode.parentElement.textContent.trim() === node.textContent.trim()
    ) {
        currentNode = currentNode.parentElement;
    }
    return currentNode;
}

function addQuoteHeaderAndWrappingElements(quoteNode) {
    let blockquoteNode = quoteNode.tagName === "BLOCKQUOTE" ? quoteNode : undefined;
    let currentNode = quoteNode;
    while (!blockquoteNode && currentNode) {
        if (currentNode.parentElement && currentNode.parentElement.tagName === "BLOCKQUOTE") {
            blockquoteNode = currentNode.parentNode;
        }
        currentNode = currentNode.parentNode;
    }
    return blockquoteNode ? [findQuoteHeaderFromBlockquote(blockquoteNode), blockquoteNode] : undefined;
}

function findQuoteHeaderFromBlockquote(blockquoteNode) {
    let quoteHeaderNode;
    let node = blockquoteNode.previousSibling;
    while (!quoteHeaderNode && node) {
        if (node.textContent && node.textContent.replace(/\s+/, "").length) {
            quoteHeaderNode = node;
        }
        node = node.previousSibling;
    }
    return quoteHeaderNode;
}

function findNodeContainingTextNodes(rootDocument, nodeWithTextNodesToMatch) {
    const spacesRegex = /\s|\n|&nbsp;/g;
    const textToMatch = nodeWithTextNodesToMatch.innerText.replace(spacesRegex, "");

    let deepestMatchingNode;
    let deepestMatchingNodeDepth = 0;
    const nodeIterator = rootDocument.createNodeIterator(
        rootDocument.body,
        NodeFilter.SHOW_ALL,
        () => NodeFilter.FILTER_ACCEPT
    );
    let node;
    while ((node = nodeIterator.nextNode())) {
        node.depth = node?.parentNode.depth + 1 || 0;
        if (
            node.innerText &&
            node.innerText.replace(spacesRegex, "").includes(textToMatch) &&
            node.depth > deepestMatchingNodeDepth
        ) {
            deepestMatchingNode = node;
        }
    }
    return deepestMatchingNode;
}

/** Find the last top level Blockquote. */
function findLastTopLevelBlockquote(htmlDoc) {
    const xpath = `(//blockquote[not(ancestor::blockquote)])[last()]`;
    const node = htmlDoc?.evaluate(
        xpath,
        htmlDoc.body,
        null,
        XPathResult.FIRST_ORDERED_NODE_TYPE,
        null
    )?.singleNodeValue;
    return node && [node];
}
