import { chunk } from "@bluemind/commons/utils/array";
import { flatMap, map } from "@bluemind/commons/utils/collection";
import { ItemFlag } from "@bluemind/core.container.api";
import { inject } from "@bluemind/inject";
import { messageUtils, loadingStatusUtils } from "@bluemind/mail";
import { extractFolderUid } from "@bluemind/mbox";

import { buildSearchQuery } from "../../components/MailSearch/SearchHelper";
import { ConversationListFilter, SortOrder } from "../conversationList";
import { FolderAdaptor } from "../folders/helpers/FolderAdaptor";

const { MessageAdaptor, getLeafParts } = messageUtils;
const { LoadingStatus } = loadingStatusUtils;

export const MAX_CHUNK_SIZE = 500;

export default {
    multipleDeleteById(messages) {
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, ({ itemsId }, folder) => api(folder).multipleDeleteById(itemsId));
        return Promise.all(requests);
    },
    deleteFlag(messages, mailboxItemFlag) {
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, ({ itemsId }, folder) => api(folder).deleteFlag({ itemsId, mailboxItemFlag }));
        return Promise.all(requests);
    },
    addFlag(messages, mailboxItemFlag) {
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, ({ itemsId }, folder) => api(folder).addFlag({ itemsId, mailboxItemFlag }));
        return Promise.all(requests);
    },
    async multipleGetById(messages) {
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, async ({ itemsId, folderRef }, folderUid) => {
            const items = flatMap(
                await Promise.all(
                    chunk(itemsId, MAX_CHUNK_SIZE).map(chunkedIds => api(folderUid).multipleGetById(chunkedIds))
                )
            );
            return items.map(item => itemToMessage(item, folderRef));
        });
        return flatMap(await Promise.all(requests));
    },
    async getForUpdate(message) {
        const remoteMessage = await api(message.folderRef.uid).getForUpdate(message.remoteRef.internalId);
        return MessageAdaptor.fromMailboxItem(remoteMessage, message.folderRef);
    },
    move(messages, destination) {
        const destinationUid = destination.remoteRef.uid;
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, ({ itemsId }, sourceUid) => importApi(sourceUid, destinationUid).move(itemsId));
        return Promise.all(requests);
    },
    sortedIds(filter, sort, folder) {
        return api(folder.remoteRef.uid).sortedIds(toSortDescriptor(filter, sort));
    },

    async search({ pattern, folder, deep }, filter, sort, currentFolder) {
        const payload = {
            query: searchQuery(pattern, filter, folder && folder.uid, deep),
            sort: toSearchSort(sort)
        };
        const { results, hasMoreResults } = await folderApi(currentFolder.mailboxRef.uid).searchItems(payload);
        if (!results) {
            return {};
        }
        const resultKeys = new Set();
        const filteredResults = [];
        results.forEach(({ itemId, containerUid }) => {
            const resultKey = `${itemId}@${containerUid}`;
            if (!resultKeys.has(resultKey)) {
                const folderRef = FolderAdaptor.toRef(extractFolderUid(containerUid));
                filteredResults.push({ id: itemId, folderRef });
                resultKeys.add(resultKey);
            }
        });
        return { results: filteredResults || [], hasMoreResults };
    },
    fetchComplete(message) {
        return api(message.folderRef.uid).fetchComplete(message.remoteRef.imapUid);
    },
    removeParts(message) {
        getLeafParts(message.structure).forEach(
            part => part.address && api(message.folderRef.uid).removePart(part.address, { keepalive: true })
        );
    },
    multipleUnexpungeById(messages) {
        const byFolder = groupByFolder(messages);
        const requests = map(byFolder, ({ itemsId }, folder) => {
            return api(folder).multipleUnexpungeById(itemsId);
        });
        return Promise.all(requests);
    }
};

function itemToMessage(item, folderRef) {
    try {
        if (!item.value.body.structure) {
            throw new Error(`Message body structure is null - item uid: ${item.uid} date: ${item?.value?.body?.date}`);
        }
        return MessageAdaptor.fromMailboxItem(item, folderRef);
    } catch (error) {
        console.error(error);
        return { loading: LoadingStatus.ERROR };
    }
}

function searchQuery(pattern, filter, folderUid, deep) {
    const query = buildSearchQuery(pattern, filter, folderUid, deep);
    return {
        ...query,
        maxResults: MAX_CHUNK_SIZE
    };
}

function api(folderUid) {
    return inject("MailboxItemsPersistence", folderUid);
}

function folderApi(mailboxUid) {
    return inject("MailboxFoldersPersistence", mailboxUid);
}

function importApi(sourceUid, destinationUid) {
    return inject("ItemsTransferPersistence", sourceUid, destinationUid);
}

function groupByFolder(messages) {
    return messages.reduce((byFolder, message) => {
        if (!byFolder[message.folderRef.uid]) {
            byFolder[message.folderRef.uid] = { itemsId: [], folderRef: message.folderRef };
        }
        byFolder[message.folderRef.uid].itemsId.push(message.remoteRef.internalId);
        return byFolder;
    }, {});
}

function toSearchSort(sort) {
    const searchSort = { criteria: [{ field: null, order: sort.order === SortOrder.ASC ? "Asc" : "Desc" }] };
    switch (sort.field) {
        case "date":
        case "size":
            searchSort.criteria[0].field = sort.field;
            break;
        case "subject":
            searchSort.criteria[0].field = "subject_kw";
            break;
        case "sender":
            searchSort.criteria[0].field = "headers.from";
            break;
    }
    return searchSort;
}

function toSortDescriptor(filter, sort) {
    const sortDescriptor = {
        fields: [
            {
                column: sort.field === "date" ? "internal_date" : sort.field,
                dir: sort.order === SortOrder.ASC ? "Asc" : "Desc"
            }
        ],
        filter: { must: [], mustNot: [] }
    };
    switch (filter) {
        case ConversationListFilter.ALL:
            sortDescriptor.filter.mustNot.push(ItemFlag.Deleted);
            break;
        case ConversationListFilter.UNREAD:
            sortDescriptor.filter.mustNot.push(ItemFlag.Seen);
            sortDescriptor.filter.mustNot.push(ItemFlag.Deleted);
            break;
        case ConversationListFilter.FLAGGED:
            sortDescriptor.filter.must.push(ItemFlag.Important);
            sortDescriptor.filter.mustNot.push(ItemFlag.Deleted);
            break;
        case ConversationListFilter.DELETED:
            sortDescriptor.filter.must.push(ItemFlag.Deleted);
            break;
    }
    return sortDescriptor;
}
