const MESSAGE_QUERY_PARTS = {
    mailbox: "m",
    folder: "d",
    search: "s",
    filter: "f",
    sort: "o"
};

export default {
    parse(path: string) {
        return pathToMessageQuery(MESSAGE_QUERY_PARTS, path);
    },
    build(path: string, params: unknown) {
        const old = pathToMessageQuery(MESSAGE_QUERY_PARTS, path);
        return messageQueryToPath(MESSAGE_QUERY_PARTS, Object.assign(old, params));
    }
};

function pathToMessageQuery(parts: Record<string, string>, path: string) {
    const patterns = Object.keys(parts).map(partKey => partToRegexp(parts[partKey]));
    const regexp = new RegExp("^" + patterns.join("") + "$");
    if (regexp.test(path)) {
        return extractGroups(regexp.exec(path), Object.keys(parts));
    } else {
        return {};
    }
}

function extractGroups(matches: RegExpExecArray | null, groups: string[]) {
    const result: Record<string, object | string> = {};
    if (matches != null) {
        groups.forEach((key, index) => (result[key] = matches[(index + 1) * 2]));
    }
    return result;
}

function partToRegexp(part: string) {
    return "(\\/?\\." + part + "\\/(.*?))?";
}

function messageQueryToPath(parts: Record<string, string>, messageQuery: Record<string, object | string>) {
    return Object.keys(parts)
        .map(part => messageQuery[part] && "." + parts[part] + "/" + toString(messageQuery[part]))
        .filter(Boolean)
        .join("/");
}

function toString(param: object | string) {
    if (typeof param === "object") {
        return Object.values(param).join(" ");
    }
    return param;
}
