import { markRaw } from "vue";
import { mapGetters } from "vuex";

import { AlertTypes, DispatchAlertMixin } from "@bluemind/bluetiful";
import ContentNames from "@bluemind/bluetiful/components/richeditor/ContentNames";
import i18n from "@bluemind/i18n";
import { mailTipUtils, signatureUtils } from "@bluemind/mail";

import { DEBOUNCED_SAVE_MESSAGE, TOGGLE_SIGNATURE } from "~/actions";
import CorporateSignatureAlert from "~/components/MailAlerts/CorporateSignatureAlert";
import { SIGNATURE, DISCLAIMER } from "~/getters";
import { SET_MAIL_TIPS, SIGNATURE_TOGGLED } from "~/mutations";

const { CORPORATE_SIGNATURE_PLACEHOLDER } = signatureUtils;
const { sameSignature } = signatureUtils;
const { getMailTipContext } = mailTipUtils;

const corporateSignatureGotInserted = {
    uid: "CORPORATE_SIGNATURE",
    renderer: { component: markRaw(CorporateSignatureAlert), props: { name: "mail.CORPORATE_SIGNATURE_INSERTED" } },
    type: AlertTypes.INFO
};
const corporateSignatureGotRemoved = {
    uid: "CORPORATE_SIGNATURE",
    renderer: { component: markRaw(CorporateSignatureAlert), props: { name: "mail.CORPORATE_SIGNATURE_REMOVED" } },
    type: AlertTypes.INFO
};

/** This mixin should only be used by MailComposerContent component. */
export default {
    mixins: [DispatchAlertMixin],
    props: {
        message: {
            type: Object,
            required: true
        }
    },
    computed: {
        ...mapGetters("mail", {
            $_SignatureMixin_disclaimer: DISCLAIMER,
            signature: SIGNATURE
        }),
        signByDefault() {
            return this.$store.state.settings.insert_signature === "true";
        }
    },
    data() {
        return { count: 0 };
    },

    async created() {
        this.$_SignatureMixin_refreshMailTips();
    },
    watch: {
        signature: {
            handler: async function (updatedSignature, previous) {
                const updatedSignatureIsValid = !!updatedSignature?.html?.innerHTML;
                const previousSignatureIsValid = !!previous?.html?.innerHTML;
                if (updatedSignatureIsValid || previousSignatureIsValid) {
                    const editorRef = await this.waitForEditor();

                    if (updatedSignatureIsValid && updatedSignature.isCorporate) {
                        editorRef.setCorporateSignature({
                            html: updatedSignature?.html,
                            options: {
                                readOnly: true,
                                movable: updatedSignature.usePlaceholder,
                                tooltip: updatedSignature.usePlaceholder
                                    ? i18n.global.t("mail.compose.corporate_signature.use_placeholder")
                                    : i18n.global.t("mail.compose.corporate_signature.read_only")
                            }
                        });
                    }

                    if (!previousSignatureIsValid) {
                        const alreadyHasPersonalSig = this.containsPersonalSignature(editorRef);
                        const alreadyHasCorpoSig = this.containsCorporateSignature(editorRef);
                        const hasCorpoSigPlaceholder = this.containsCorporateSignaturePlaceholder(editorRef);
                        if (!alreadyHasPersonalSig && !alreadyHasCorpoSig && !hasCorpoSigPlaceholder) {
                            this.insertSignature(editorRef, updatedSignature);
                        } else if (hasCorpoSigPlaceholder) {
                            this.replaceCorpoSigPlaceholder(editorRef, updatedSignature);
                        }
                    } else if (!updatedSignatureIsValid) {
                        this.removeSignatures(editorRef);
                    } else if (!sameSignature(previous, updatedSignature)) {
                        this.replaceSignature(editorRef, updatedSignature);
                    }

                    this.notifySignatureChange(previous, updatedSignature);
                    this.insertDisclaimer(editorRef);
                    editorRef.onChange();
                }
            },
            immediate: true
        },

        "message.from": {
            async handler(value, oldValue) {
                if ((value && !oldValue) || value?.address !== oldValue?.address) {
                    this.$_SignatureMixin_refreshMailTips();
                }
            }
        },

        $_SignatureMixin_disclaimer: {
            async handler(value, oldValue) {
                if (value && !oldValue) {
                    const editorRef = await this.waitForEditor();
                    this.insertDisclaimer(editorRef);
                    editorRef.onChange();
                }
            },
            immediate: true
        }
    },
    methods: {
        $_SignatureMixin_refreshMailTips() {
            this.$execute("get-mail-tips", { context: getMailTipContext(this.message), message: this.message });
        },
        removeSignatures(editorRef) {
            editorRef.removeContent([ContentNames.SIGNATURE, ContentNames.CORPORATE_SIGNATURE]);
            this.removeCorpoSigPlaceholder(editorRef);
        },
        removeCorpoSigPlaceholder(editorRef) {
            editorRef.removeText(CORPORATE_SIGNATURE_PLACEHOLDER);
        },
        insertDisclaimer(editorRef) {
            editorRef.removeContent(ContentNames.DISCLAIMER, { removeBrAfter: false });
            editorRef.addContent(this.$_SignatureMixin_disclaimer, {
                name: ContentNames.DISCLAIMER,
                readOnly: true,
                tooltip: i18n.global.t("mail.compose.corporate_signature.read_only"),
                brAfter: false
            });
        },
        insertSignature(editorRef, signature) {
            if (signature.html?.innerHTML) {
                let readOnly = false;
                let name = ContentNames.SIGNATURE;
                let tooltip;
                if (signature.isCorporate) {
                    readOnly = true;
                    name = ContentNames.CORPORATE_SIGNATURE;
                    tooltip = signature.usePlaceholder
                        ? i18n.global.t("mail.compose.corporate_signature.use_placeholder")
                        : i18n.global.t("mail.compose.corporate_signature.read_only");
                }
                editorRef.addContent(signature.html, {
                    name,
                    readOnly,
                    insertBefore: ContentNames.PREVIOUS_MESSAGE,
                    movable: signature.usePlaceholder,
                    tooltip
                });
            }
        },
        replaceSignature(editorRef, updatedSignature) {
            const tooltip = updatedSignature.usePlaceholder
                ? i18n.global.t("mail.compose.corporate_signature.use_placeholder")
                : i18n.global.t("mail.compose.corporate_signature.read_only");
            editorRef.replaceContent(
                [ContentNames.SIGNATURE, ContentNames.CORPORATE_SIGNATURE],
                updatedSignature?.html,
                updatedSignature?.isCorporate ? ContentNames.CORPORATE_SIGNATURE : ContentNames.SIGNATURE,
                { readOnly: updatedSignature?.isCorporate, movable: updatedSignature.usePlaceholder, tooltip }
            );
        },
        replaceCorpoSigPlaceholder(editorRef, updatedSignature) {
            const tooltip = updatedSignature.usePlaceholder
                ? i18n.global.t("mail.compose.corporate_signature.use_placeholder")
                : i18n.global.t("mail.compose.corporate_signature.read_only");
            editorRef.replaceTextByContent(
                CORPORATE_SIGNATURE_PLACEHOLDER,
                updatedSignature?.html,
                ContentNames.CORPORATE_SIGNATURE,
                {
                    readOnly: true,
                    movable: updatedSignature.usePlaceholder,
                    tooltip
                }
            );
        },
        notifySignatureChange(removedSignature, insertedSignature) {
            if (insertedSignature?.uid && !removedSignature?.uid) {
                this.dispatchAlert(corporateSignatureGotInserted);
            }

            if (removedSignature?.uid && !insertedSignature?.uid) {
                this.dispatchAlert(corporateSignatureGotRemoved);
            }
        },
        async toggleSignature() {
            this.$store.dispatch(`mail/${TOGGLE_SIGNATURE}`);
            this.$store.dispatch(`mail/${DEBOUNCED_SAVE_MESSAGE}`, { draft: this.message });
        },
        $_SignatureMixin_resetAlerts() {
            this.removeAlert(corporateSignatureGotInserted);
            this.removeAlert(corporateSignatureGotRemoved);
        },
        containsPersonalSignature(editorRef) {
            return !!editorRef
                ?.getRootElement()
                .querySelector(`*:not(blockquote) [${signatureUtils.PERSONAL_SIGNATURE_ATTR}]`);
        },
        containsCorporateSignature(editorRef) {
            return editorRef.hasNamedContent(ContentNames.CORPORATE_SIGNATURE);
        },
        containsCorporateSignaturePlaceholder(editorRef) {
            return !!editorRef?.hasCorpoSigPlaceholder();
        },
        checkPersonalSignature(editorRef) {
            const alreadyHasPersonalSig = this.containsPersonalSignature(editorRef);
            const toggled = alreadyHasPersonalSig || this.signByDefault;
            this.$store.commit(`mail/${SIGNATURE_TOGGLED}`, toggled);
        }
    },
    unmounted() {
        this.$_SignatureMixin_resetAlerts();
        this.$store.commit(`mail/${SET_MAIL_TIPS}`, []);
    }
};
