import { computed } from "vue";

import { cloneDeep } from "@bluemind/commons/utils/lang";
import { messageUtils } from "@bluemind/mail";
import router from "@bluemind/router";
import store from "@bluemind/store";

import { SAVE_MESSAGE, SEND_MESSAGE, SCHEDULE_SEND_MESSAGE } from "~/actions";
import {
    CONVERSATIONS_ACTIVATED,
    CURRENT_CONVERSATION_METADATA,
    MY_DRAFTS,
    MY_OUTBOX,
    MY_MAILBOX_KEY
} from "~/getters";
import { SET_MESSAGE_HEADERS } from "~/mutations";
const { setHeader, removeHeader } = messageUtils;

export function useSend() {
    const myMailboxKey = computed(() => store.getters[`mail/${MY_MAILBOX_KEY}`]);
    const outbox = computed(() => store.getters[`mail/${MY_OUTBOX}`]);
    const drafts = computed(() => store.getters[`mail/${MY_DRAFTS}`]);
    const messageCompose = computed(() => store.state.mail.messageCompose);
    const conversationActivacted = computed(() => store.getters[`mail/${CONVERSATIONS_ACTIVATED}`]);
    const currentConversation = computed(() => store.state.mail.conversations.currentConversation);
    const currentConversationMetadata = computed(() => store.getters[`mail/${CURRENT_CONVERSATION_METADATA}`]);

    function sendNow(message) {
        const date = new Date();
        const headers = message.headers;
        setHeader(headers, messageUtils.MessageHeader.X_BM_DRAFT_REFRESH_DATE, date.getTime());
        removeHeader(headers, messageUtils.MessageHeader.X_BM_DRAFT_SCHEDULE);
        saveHeaders(message, headers);
        saveAsap(message);
        return sendAction(SEND_MESSAGE, message);
    }

    function sendAction(actionName, message) {
        if (!isSaved(message)) {
            saveAsap(message);
        }

        store.dispatch(`mail/${actionName}`, {
            draft: message,
            myMailboxKey: myMailboxKey.value,
            outbox: outbox.value,
            myDraftsFolder: drafts.value,
            messageCompose: cloneDeep(messageCompose.value),
            subject: message.subject
        });
        if (
            !conversationActivacted.value ||
            !currentConversation.value ||
            currentConversationMetadata.value.messages.length < 2
        ) {
            router.navigate("v:mail:home");
        }
    }
    async function saveAsap(message) {
        await store.dispatch(`mail/${SAVE_MESSAGE}`, {
            draft: message
        });
    }

    async function saveHeaders(message, headers) {
        store.commit("mail/" + SET_MESSAGE_HEADERS, { messageKey: message.key, headers });
        await saveAsap(message);
    }
    async function scheduleSend(date, message) {
        const headers = [...message.headers];
        messageUtils.setHeader(headers, messageUtils.MessageHeader.X_BM_DRAFT_SCHEDULE, date.toString());
        messageUtils.setHeader(headers, messageUtils.MessageHeader.X_BM_DRAFT_REFRESH_DATE, date.toString());
        await saveHeaders(message, headers);
        sendAction(SCHEDULE_SEND_MESSAGE, message);
    }

    function isSaved(message) {
        return Boolean(message.remoteRef.internalId);
    }

    return { sendAction, sendNow, scheduleSend };
}
