import inject from "@bluemind/inject";

import { usePhishingUtils } from "./usePhishingUtils.composable";

describe("usePhishingUtils", () => {
    describe("checkPhishing", () => {
        const { checkPhishing } = usePhishingUtils();
        const addressBookContacts = [
            { value: { mail: "test.phishing.one@dev.bluemind.net", formatedName: "Test One" } },
            { value: { mail: "test.phishing.two@dev.bluemind.net", formatedName: "Test Two" } },
            { value: { mail: "", formatedName: "Test Three" } }
        ];
        inject.register({
            provide: "AddressBooksPersistence",
            factory: () => ({
                search: (request: { query: string }) => {
                    const pattern = /value:\((.*?)\)/.exec(request.query)?.[1];
                    const values = addressBookContacts.filter(abc => abc.value.formatedName === pattern);
                    return { total: values.length, values };
                }
            })
        });
        test("Unknown contact is not phishing", async () => {
            const contact = { address: "test.phishing.four@dev.bluemind.net", dn: "Test Four" };
            const isPhishing = await checkPhishing(contact);
            expect(isPhishing).toBeFalsy();
        });
        test("Contact without a display name is not phishing", async () => {
            const contact = { address: "test.phishing@dev.bluemind.net", dn: "" };
            const isPhishing = await checkPhishing(contact);
            expect(isPhishing).toBeFalsy();
        });
        test("Contact existing in address books with a different email address is phishing", async () => {
            const contact = { address: "when.a.phish@is.not.a.fish", dn: "Test One" };
            const isPhishing = await checkPhishing(contact);
            expect(isPhishing).toBeTruthy();
        });
        test("Given contact has an address but not the equivalent one in address books: its phishing", async () => {
            const contact = { address: "when.a.phish@is.not.a.fish", dn: "Test Three" };
            const isPhishing = await checkPhishing(contact);
            expect(isPhishing).toBeTruthy();
        });
    });
});
