import LuceneQueryParser, { type AST } from "lucene";

import { toArray } from "@bluemind/commons/utils/array";

import { ConversationListFilter } from "../../../store/conversationList";

import { escapeTerm, unescapeTerm } from "./escape";
import PATTERN_KEYWORDS, { RECORD_QUERY_FIELDS, QUERY_FIELDS } from "./Keywords";
import buildQuery from "./queryBuilder";

type Node = {
    left?: Node;
    right?: Node;
    quoted: boolean;
    field: string;
    term: string;
};
type ParsedSearchPattern = {
    [key: string]: string | string[];
};

type ParsedQuery = {
    pattern?: string;
    folder?: string;
    deep?: boolean;
};

export function buildSearchQuery(
    queryPattern: string,
    filter: ConversationListFilter,
    folderUid: string,
    deep?: boolean
) {
    const parsedPattern = SearchHelper.parseSearchPattern(queryPattern);
    return buildQuery(parsedPattern, filter, folderUid, deep);
}

export function parseSearchPattern(pattern: string): ParsedSearchPattern {
    if (!pattern) {
        return {};
    }

    const result: ParsedSearchPattern = {};
    const termsOnly: string[] = [];

    try {
        const rootNode = LuceneQueryParser.parse(pattern) as Node;

        const setField = (field: string, newValue: string) => {
            if (!result[field]) {
                result[field] = newValue;
            } else {
                result[field] = toArray(result[field]);
                result[field].push(newValue);
            }
        };
        const nodeFunction = (node: Node, exit?: boolean): boolean => {
            const field = node.field;

            if (exit) {
                return true;
            }

            if (field === PATTERN_KEYWORDS.CONTENT && node.right) {
                const right = node.right;
                const content = LuceneQueryParser.toString(right as AST);
                termsOnly.push(right.quoted ? unescapeTerm(content) : content);
            } else if (Object.values(PATTERN_KEYWORDS).includes(field)) {
                if (node.term) {
                    const escaped = node.quoted ? unescapeTerm(node.term) : node.term;
                    setField(field, escaped);
                } else if (node.left || node.right) {
                    result[field] = extractAllTerms(node);
                } else {
                    const value =
                        LuceneQueryParser.toString(node as AST)
                            .split(":")
                            .pop() ?? "";
                    const escaped = node.quoted ? unescapeTerm(value) : value;
                    setField(field, escaped);
                }
            } else if (field) {
                const value = LuceneQueryParser.toString(node as AST);
                termsOnly.push(node.quoted ? unescapeTerm(value) : value);
            }

            return !!field;
        };
        walkLuceneTree(rootNode, nodeFunction);

        if (termsOnly.length > 0) {
            result.content = termsOnly.join(" ");
        }
    } catch {
        result.content = pattern;
    }

    return result;
}

function extractAllTerms(node: Node | null): string[] {
    if (!node) return [];

    const terms: string[] = [];

    if (node.term) {
        const value = node.term;
        terms.push(node.quoted ? unescapeTerm(value) : value);
        return terms;
    }

    if (node.left || node.right) {
        if (node.left) {
            terms.push(...extractAllTerms(node.left));
        }
        if (node.right) {
            terms.push(...extractAllTerms(node.right));
        }
        return terms;
    }

    return terms;
}

const folderParser = (node: Node, result: ParsedQuery): void => {
    if (node.field === "in") {
        result.folder = node.term;
    }
};

const deepParser = (node: Node, result: ParsedQuery): void => {
    if (node.field === "is") {
        result.deep = result.deep || node.term === "deep";
    }
};

const queryParsers: Array<(node: Node, result: ParsedQuery) => void> = [folderParser, deepParser];

function parseQuery(expression: string): ParsedQuery {
    const result: ParsedQuery = {};
    try {
        if (expression) {
            const rootNode = LuceneQueryParser.parse(expression) as Node;
            if (rootNode) {
                result.pattern = rootNode.left?.term;
                const nodeFunction = (node: Node): boolean => {
                    queryParsers.forEach(parser => parser(node, result));
                    return false;
                };
                walkLuceneTree(rootNode, nodeFunction);
            }
        }
        return result;
    } catch {
        return { pattern: expression };
    }
}

function isSameSearch(
    previousPattern: string,
    pattern: string,
    previousFolderKey: string,
    folderKey: string,
    isDeep?: boolean,
    previousIsDeep?: boolean
): boolean {
    const isSamePattern = pattern === previousPattern;
    const isSameFolder = folderKey === previousFolderKey;
    const isSameDepth = !!isDeep === !!previousIsDeep;
    const isAllFoldersAgain = !folderKey && !previousFolderKey;
    return isSamePattern && isSameDepth && (isSameFolder || isAllFoldersAgain);
}

function walkLuceneTree(node: Node, nodeFunction: (node: Node, exit?: boolean) => boolean, exit?: boolean): void {
    const toExit = nodeFunction(node, exit) ?? undefined;

    if (node.left) {
        walkLuceneTree(node.left, nodeFunction, toExit);

        if (node.right) {
            walkLuceneTree(node.right, nodeFunction, toExit);
        }
    }
}

export const SearchHelper = {
    unescapeTerm,
    escapeTerm,
    parseQuery,
    parseSearchPattern,
    isSameSearch,
    PATTERN_KEYWORDS,
    RECORD_QUERY_FIELDS,
    QUERY_FIELDS
};

export default SearchHelper;
