import { computed, h } from "vue";

import { TooLargeBox } from "@bluemind/business-components";
import { useCommand } from "@bluemind/command";
import app from "@bluemind/global.app";
import i18n from "@bluemind/i18n";
import { partUtils, attachmentUtils, fileUtils } from "@bluemind/mail";
import store from "@bluemind/store";
import UUIDGenerator from "@bluemind/uuid";

import { ADD_ATTACHMENT, DEBOUNCED_SAVE_MESSAGE } from "~/actions";

const { createFromFile: createPartFromFile } = partUtils;
const { create } = attachmentUtils;
const { FileStatus } = fileUtils;

async function addAttachments({ files, message, maxSize }) {
    const filesBlob = [...files];

    const totalSize = filesBlob.reduce((total, blob) => total + blob.size, message.size);
    if (totalSize > maxSize) {
        renderTooLargeFilesModal.call(this, files, maxSize);
        return;
    }

    const promises = filesBlob.map(fileBlob => {
        const part = createPartFromFile({
            address: UUIDGenerator.generate(),
            name: fileBlob.name,
            type: fileBlob.type,
            size: fileBlob.size
        });
        const attachment = create(part, FileStatus.NOT_LOADED);

        return store.dispatch(`mail/${ADD_ATTACHMENT}`, {
            message,
            attachment,
            content: fileBlob
        });
    });
    await Promise.all(promises);
    await store.dispatch(`mail/${DEBOUNCED_SAVE_MESSAGE}`, {
        draft: message,
        messageCompose: store.state.mail.messageCompose
    });
}

export default {
    commands: {
        addAttachments: function (payload) {
            const maxSize = store.state.mail.messageCompose.maxMessageSize;
            return addAttachments.call(this, { maxSize, ...payload });
        }
    }
};
export function useAddAttachmentsCommand() {
    const maxSize = computed(() => store.state.mail.messageCompose.maxMessageSize);
    return {
        maxSize,
        execAddAttachments: useCommand("addAttachments", function (payload) {
            return addAttachments.call(this, { maxSize: maxSize.value, ...payload });
        })
    };
}

async function renderTooLargeFilesModal(files, sizeLimit) {
    const content = h(TooLargeBox, { sizeLimit, attachmentsCount: files.length });
    const props = {
        content,
        size: "sm",
        title: i18n.global.t("mail.actions.attach.too_large", files.length),
        okLabel: i18n.global.t("common.got_it"),
        okOnly: true,
        okVariant: "outline-accent"
    };

    await app.config.globalProperties.$modalConfirm.confirm(props);
}
