import { describe, it, expect, beforeEach, vi } from "vitest";

import { MailboxItem, MailboxItemFlag } from "@bluemind/backend.mail.api";
import { ItemFlag, ItemValue } from "@bluemind/core.container.api";

import MailboxItemEntity from "./MailboxItemEntity";

vi.mock("@bluemind/commons.light/utils/lang", () => ({
    cloneDeep: vi.fn(obj => JSON.parse(JSON.stringify(obj)))
}));

describe("MailboxItemEntity", () => {
    let mockItem: ItemValue<MailboxItem & { conversationId: string }>;
    let entity: MailboxItemEntity;
    beforeEach(() => {
        mockItem = {
            flags: [ItemFlag.Seen],
            value: {
                conversationId: "convid",
                flags: [MailboxItemFlag.System.Seen as MailboxItemFlag],
                body: {}
            }
        };
        entity = new MailboxItemEntity(mockItem);
    });

    describe("constructor", () => {
        it("should clone the input item", () => {
            const originalItem = { ...mockItem };
            const newEntity = new MailboxItemEntity(mockItem);

            mockItem.uid = "modified-uid";
            expect(newEntity.toItem().uid).toBe(originalItem.uid);
        });

        it("should handle item with undefined flags", () => {
            const itemWithoutFlags = {
                ...mockItem,
                flags: undefined,
                value: { ...mockItem.value, flags: undefined }
            };

            expect(() => new MailboxItemEntity(itemWithoutFlags)).not.toThrow();
        });
    });

    describe("addFlag", () => {
        it("should add a new flag to both item.value.flags and item.flags", () => {
            const initialValueFlags = entity.toItem().value.flags?.length || 0;
            const initialItemFlags = entity.toItem().flags?.length || 0;

            entity.addFlag(MailboxItemFlag.System.Flagged);
            const result = entity.toItem();

            expect(result.value.flags).toContain(MailboxItemFlag.System.Flagged);
            expect(result.flags).toContain(ItemFlag.Important);
            expect(result.value.flags?.length).toBe(initialValueFlags + 1);
            expect(result.flags?.length).toBe(initialItemFlags + 1);
        });

        it("should not add duplicate flags", () => {
            entity.addFlag(MailboxItemFlag.System.Seen);
            entity.addFlag(MailboxItemFlag.System.Seen);

            const result = entity.toItem();
            const seenFlags = result.value.flags?.filter(f => f === (MailboxItemFlag.System.Seen as MailboxItemFlag));

            expect(seenFlags?.length).toBe(1);
        });

        it("should handle item with initially undefined flags", () => {
            const itemWithoutFlags = {
                ...mockItem,
                flags: undefined,
                value: { ...mockItem.value, flags: undefined }
            };

            const newEntity = new MailboxItemEntity(itemWithoutFlags);
            newEntity.addFlag(MailboxItemFlag.System.Flagged);

            const result = newEntity.toItem();
            expect(result.value.flags).toContain(MailboxItemFlag.System.Flagged);
            expect(result.flags).toContain(ItemFlag.Important);
        });

        it("should return the entity instance for chaining", () => {
            const result = entity.addFlag(MailboxItemFlag.System.Flagged);
            expect(result).toBe(entity);
        });

        it("should handle custom flags (non-system flags)", () => {
            const customFlag = "CustomFlag";
            entity.addFlag(customFlag);

            const result = entity.toItem();
            expect(result.value.flags).toContain(customFlag);
            expect(result.flags).toContain(customFlag);
        });
    });

    describe("removeFlag", () => {
        beforeEach(() => {
            entity.addFlag(MailboxItemFlag.System.Flagged);
            entity.addFlag(MailboxItemFlag.System.Deleted);
        });

        it("should remove existing flag from both item.value.flags and item.flags", () => {
            entity.removeFlag(MailboxItemFlag.System.Flagged);
            const result = entity.toItem();

            expect(result.value.flags).not.toContain(MailboxItemFlag.System.Flagged);
            expect(result.flags).not.toContain(ItemFlag.Important);
        });

        it("should handle removal of non-existing flag gracefully", () => {
            const nonExistingFlag = "NonExisting";

            expect(() => entity.removeFlag(nonExistingFlag)).not.toThrow();
        });

        it("should handle item with undefined flags", () => {
            const itemWithoutFlags = {
                ...mockItem,
                flags: undefined,
                value: { ...mockItem.value, flags: undefined }
            };

            const newEntity = new MailboxItemEntity(itemWithoutFlags);
            expect(() => newEntity.removeFlag(MailboxItemFlag.System.Seen)).not.toThrow();
        });

        it("should return the entity instance for chaining", () => {
            const result = entity.removeFlag(MailboxItemFlag.System.Seen);
            expect(result).toBe(entity);
        });

        it("should handle flags with backslashes correctly", () => {
            const flagWithBackslash = "\\Seen";
            entity.addFlag(flagWithBackslash);
            entity.removeFlag(MailboxItemFlag.System.Seen);

            const result = entity.toItem();
            expect(result.value.flags).not.toContain(flagWithBackslash);
        });
    });

    describe("method chaining", () => {
        it("should allow chaining addFlag and removeFlag calls", () => {
            const result = entity
                .addFlag(MailboxItemFlag.System.Flagged)
                .addFlag(MailboxItemFlag.System.Deleted)
                .removeFlag(MailboxItemFlag.System.Seen)
                .toItem();

            expect(result.value.flags).toContain(MailboxItemFlag.System.Flagged);
            expect(result.value.flags).toContain(MailboxItemFlag.System.Deleted);
            expect(result.value.flags).not.toContain(MailboxItemFlag.System.Seen);
        });
    });

    describe("toItem", () => {
        it("should return the current item state", () => {
            const result = entity.toItem();
            expect(result).toBeDefined();
            expect(result.uid).toBe(mockItem.uid);
            expect(result.value.conversationId).toBe(mockItem.value.conversationId);
        });

        it("should return updated item after modifications", () => {
            entity.addFlag(MailboxItemFlag.System.Flagged);
            const result = entity.toItem();

            expect(result.value.flags).toContain(MailboxItemFlag.System.Flagged);
            expect(result.flags).toContain(ItemFlag.Important);
        });
    });

    describe("flag mapping", () => {
        it("should correctly map MailboxItemFlag.System.Seen to ItemFlag.Seen", () => {
            entity.addFlag(MailboxItemFlag.System.Seen);
            const result = entity.toItem();

            expect(result.flags).toContain(ItemFlag.Seen);
        });

        it("should correctly map MailboxItemFlag.System.Flagged to ItemFlag.Important", () => {
            entity.addFlag(MailboxItemFlag.System.Flagged);
            const result = entity.toItem();

            expect(result.flags).toContain(ItemFlag.Important);
        });

        it("should correctly map MailboxItemFlag.System.Deleted to ItemFlag.Deleted", () => {
            entity.addFlag(MailboxItemFlag.System.Deleted);
            const result = entity.toItem();

            expect(result.flags).toContain(ItemFlag.Deleted);
        });
    });

    describe("rawFlag method behavior", () => {
        it("should handle flags with backslashes in remove operations", () => {
            const flagWithBackslash = "\\Flagged";
            const flagWithoutBackslash = "Flagged";

            entity.addFlag(flagWithBackslash);
            entity.removeFlag(flagWithoutBackslash);

            const result = entity.toItem();
            expect(result.value.flags?.some(f => (f as string).includes("Flagged"))).toBe(false);
        });
    });

    describe("edge cases", () => {
        it("should handle empty flags arrays", () => {
            const emptyFlagsItem = {
                ...mockItem,
                flags: [],
                value: { ...mockItem.value, flags: [] }
            };

            const newEntity = new MailboxItemEntity(emptyFlagsItem);
            newEntity.addFlag(MailboxItemFlag.System.Seen);

            const result = newEntity.toItem();
            expect(result.value.flags).toHaveLength(1);
            expect(result.flags).toHaveLength(1);
        });

        it("should preserve other item properties", () => {
            const originalConversationId = entity.toItem().value.conversationId;

            entity.addFlag(MailboxItemFlag.System.Flagged);

            const result = entity.toItem();
            expect(result.value.conversationId).toBe(originalConversationId);
        });
    });
});
