import { ContainerSyncOptions } from "@bluemind/commons.light/model/synchronization";

import { synchronize } from "./Synchronize";

export default {
    inProgress: new Map<string, Promise<boolean>>(),
    fifo: <Required<ContainerSyncOptions>[]>[],
    async queue(options: ContainerSyncOptions) {
        this.add(options);
        await this.flush();
    },
    add(options: ContainerSyncOptions) {
        const sanitizedOptions = { priority: 0, ...options };
        const containerInQueue = find(options, this.fifo);
        if (containerInQueue && containerInQueue.priority >= sanitizedOptions.priority) {
            return;
        } else if (containerInQueue && containerInQueue.priority <= sanitizedOptions.priority) {
            containerInQueue.priority = sanitizedOptions.priority;
        } else {
            this.fifo.push({ ...sanitizedOptions });
        }
        this.fifo.sort((a, b) => b.priority - a.priority);
    },
    isInQueue(options: ContainerSyncOptions) {
        return find(options, this.fifo) !== undefined || this.inProgress.has(options.uid);
    },
    async flush() {
        if (this.inProgress.size > 0) return false;
        while (this.fifo.length > 0) {
            await this.synchronize(0);
        }
    },
    async flushOne(options: ContainerSyncOptions) {
        if (this.inProgress.has(options.uid)) return this.inProgress.get(options.uid);

        const containerInQueue = find(options, this.fifo);
        if (!containerInQueue) return;

        const index = this.fifo.indexOf(containerInQueue);

        return this.synchronize(index);
    },

    synchronize(index: number) {
        const [options] = this.fifo.splice(index, 1);
        const promise = synchronize(options);
        this.inProgress.set(options.uid, promise);
        promise.finally(() => this.inProgress.delete(options.uid));
        return promise;
    }
};

function find(
    container: ContainerSyncOptions,
    queue: Required<ContainerSyncOptions>[]
): Required<ContainerSyncOptions> | undefined {
    return queue.find(({ uid }) => container.uid === uid);
}
