/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.deferredaction.registry;

import java.time.ZonedDateTime;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.AbstractVerticle;
import net.bluemind.cluster.common.CuratorLatchManager;
import net.bluemind.cluster.common.ILeadershipHandle;
import net.bluemind.cluster.common.ILeadershipListener;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.network.topology.Topology;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;

public class DeferredActionExecution extends AbstractVerticle {

	public static final long PERIOD = TimeUnit.MINUTES.toMillis(5);
	private static final Executor executor = Executors
			.newSingleThreadExecutor(new DefaultThreadFactory("deferred-action"));
	private static final Logger logger = LoggerFactory.getLogger(DeferredActionExecution.class);
	private static final String DEFERRED_ACTION_LATCH_NAME = "deferred-action-latch";
	private ILeadershipHandle leadership;

	@Override
	public void start() throws Exception {
		VertxPlatform.executeBlockingTimer(vertx, 1000, this::startImpl);
	}

	private void startImpl(long unused) {
		// We wait until topology is available, to prevent ugly thread blocked in
		// vertx event-loop
		if (!Topology.getIfAvailable().isPresent()) {
			VertxPlatform.executeBlockingTimer(vertx, 2000, this::startImpl);
			return;
		}

		this.leadership = CuratorLatchManager.startLatch(DEFERRED_ACTION_LATCH_NAME, new ILeadershipListener() {

			@Override
			public void lostLeadership(ISeppuku seppukuHandle) {
				// It's safer to restart to ensure no deferred action will run twice if we don't
				// have leadership in multi-core context
				if (!Topology.get().singleCore()) {
					logger.info("Leadership to run deferred action lost, restarting core.");
					seppukuHandle.commit();
				} else {
					logger.info("Leadership to run deferred action lost.");
				}
			}

			@Override
			public void obtainedLeadership() {
				logger.info("Leadership to run deferred action gained.");
			}
		});

		vertx.setPeriodic(PERIOD, this::execute);
	}

	private void execute(Long timerId) {
		if (StateContext.getState() == SystemState.CORE_STATE_RUNNING && leadership.isLeader()) {
			executor.execute(() -> {
				DeferredActionPluginLoader.executors.forEach(exe -> {
					logger.debug("Executing deferred action executor {}", exe.getSupportedActionId());
					try {
						exe.create().execute(ZonedDateTime.now().plusMinutes(5));
					} catch (Exception e) {
						logger.error("Error while executing deferred actions", e);
					}
				});
			});
		}
	}

}
