package net.bluemind.system.persistence.hot;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import javax.sql.DataSource;

import net.bluemind.core.api.DataSourceType;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.hotupgrade.repository.IHotUpgradeTaskStore;
import net.bluemind.repository.provider.IStandaloneFactory;
import net.bluemind.system.api.hot.upgrade.HotUpgradeProgress;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTask;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskExecutionMode;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskFilter;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskStatus;

public class HotUpgradeTaskStore extends JdbcAbstractStore implements IHotUpgradeTaskStore {

	private static final Creator<HotUpgradeTask> CREATOR = rs -> new HotUpgradeTask();

	public static class Factory implements IStandaloneFactory<IHotUpgradeTaskStore> {

		@Override
		public Class<IHotUpgradeTaskStore> factoryClass() {
			return IHotUpgradeTaskStore.class;
		}

		@Override
		public DataSourceType targetRepositoryType() {
			return DataSourceType.POSTGRESQL;
		}

		@Override
		public IHotUpgradeTaskStore instance(BmContext context) throws ServerFault {
			return new HotUpgradeTaskStore(context.getDataSource());
		}

	}

	public HotUpgradeTaskStore(DataSource pool) {
		super(pool);
		Objects.requireNonNull(pool, "datasource must not be null");
	}

	@Override
	public HotUpgradeTask create(HotUpgradeTask task) throws ServerFault {
		String query = "insert into t_hot_upgrade_task (" + HotUpgradeTaskColumns.COLUMNS.names() + ") values ("
				+ HotUpgradeTaskColumns.COLUMNS.values() + ")";

		String status = task.status != null ? task.status.name() : HotUpgradeTaskStatus.PLANNED.name();
		return doOrFail(() -> {
			task.id = insertWithSerial(query,
					new Object[] { task.operation, task.parameters, status, task.failure, toTimestamp(task.createdAt),
							toTimestamp(task.updatedAt), task.executionMode.name(), task.retryCount,
							task.retryDelaySeconds, task.reportFailure, JsonUtils.asString(task.events), task.mandatory,
							task.upgraderId });

			return task;
		});
	}

	@Override
	public HotUpgradeTask get(int id) throws SQLException {
		String query = "select id, " + HotUpgradeTaskColumns.COLUMNS.names() + " from t_hot_upgrade_task where id = ?";
		return unique(query, CREATOR, HotUpgradeTaskColumns.populator(), new Object[] { id });
	}

	@Override
	public List<HotUpgradeTask> list(HotUpgradeTaskFilter filter) throws SQLException {
		List<String> conditions = new ArrayList<>();
		List<Object> parameters = new ArrayList<>();
		if (filter.operation() != null) {
			conditions.add("operation = ?");
			parameters.add(filter.operation());
		}
		if (!filter.getStatuses().isEmpty()) {
			conditions.add("status = ANY(?::enum_hot_upgrade_task_status[])");
			parameters.add(filter.getStatuses().stream().map(HotUpgradeTaskStatus::name).toArray(String[]::new));
		}
		if (filter.onlyRetryable()) {
			conditions.add("(failure < retry_count or retry_count = 0)");
		}
		if (filter.onlyReady()) {
			conditions.add("(updated_at + INTERVAL '1 second' * retry_delay <= ? "
					+ "or status = 'PLANNED'::enum_hot_upgrade_task_status or mandatory = true)");
			parameters.add(now());
		}
		if (filter.onlyMandatory()) {
			conditions.add("mandatory = true");
		}
		if (!filter.getMode().isEmpty()) {
			conditions.add("execution_mode = ANY(?::enum_hot_upgrade_execution_mode[])");
			parameters.add(filter.getMode().stream().map(HotUpgradeTaskExecutionMode::name).toArray(String[]::new));
		}
		String whereClause = String.join(" and ", conditions);
		String query = "select id, " + HotUpgradeTaskColumns.COLUMNS.names() + " FROM t_hot_upgrade_task "
				+ (!whereClause.isEmpty() ? "where " + whereClause : "") + " ORDER BY upgrader_id";
		return select(query, CREATOR, HotUpgradeTaskColumns.populator(), parameters.toArray());
	}

	@Override
	public List<HotUpgradeProgress> progress() throws SQLException {
		String query = "select status, count(*) as count, max(updated_at) as last_updated_at from t_hot_upgrade_task group by status";
		return select(query, rs -> new HotUpgradeProgress(), HotUpgradeProgressColumns.populator(), new Object[0]);
	}

	@Override
	public int updateStatus(HotUpgradeTask task) throws SQLException {
		String query = "update t_hot_upgrade_task set status = ?::enum_hot_upgrade_task_status, failure = ?, eventlog = ?::jsonb, updated_at = ? where id = ?";
		return update(query,
				new Object[] { task.status.name(), task.failure, JsonUtils.asString(task.events), now(), task.id });
	}

	private Timestamp toTimestamp(Date date) {
		return date == null ? now() : Timestamp.from(date.toInstant());
	}

	private Timestamp now() {
		return Timestamp.from(Instant.now());
	}
}
