/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.restore.orphans;

import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.core.type.TypeReference;

import io.vertx.core.json.JsonObject;
import net.bluemind.authentication.api.APIKey;
import net.bluemind.authentication.repository.IAPIKeyStore;
import net.bluemind.core.api.VersionInfo;
import net.bluemind.core.backup.continuous.DataElement;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.repository.provider.RepositoryProvider;

public class RestoreApiKeys {
	private final Pattern REGEX_PRE_55_UID = Pattern
			.compile("^(.+)-([0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12})$");

	private static final ValueReader<ItemValue<APIKey>> scReader = JsonUtils
			.reader(new TypeReference<ItemValue<APIKey>>() {
			});

	public void restore(IServerTaskMonitor monitor, List<DataElement> apiKeys) {
		monitor.log("Restore " + apiKeys.size() + " API keys");
		for (var de : apiKeys) {
			if (de.payload == null) {
				// Create or delete API key must have a payload
				continue;
			}
			ItemValue<APIKey> iv = scReader.read(de.payload);

			if (de.isDelete()) {
				deleteApiKey(monitor, produceByVersion(de), iv);
			} else if (de.key.operation.equals(RecordKey.Operation.CREATE.name())
					|| de.key.operation.equals(RecordKey.Operation.UPDATE.name())) {
				createApiKey(monitor, iv);
			}
		}
	}

	private VersionInfo produceByVersion(DataElement de) {
		JsonObject parsed = new JsonObject(new String(de.payload));
		return Optional.ofNullable(parsed.getString("producedBy")).map(VersionInfo::create)
				.orElseGet(() -> VersionInfo.create("0.0.0"));
	}

	private void deleteApiKey(IServerTaskMonitor monitor, VersionInfo producedByVersion, ItemValue<APIKey> iv) {
		if (!producedByVersion.greaterThanOrEquals(VersionInfo.create("5.5.0"))) {
			monitor.log("Delete API key: " + iv + " - pre BM 5.5 format");
			deleteApiKeyPre55(monitor, iv);
			return;
		}

		// BM 5.5+, kafka ItemValue UID == uid
		monitor.log("Delete API key: " + iv);

		APIKey apiKey = RepositoryProvider.instance(IAPIKeyStore.class,
				ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext()).get(iv.uid);
		if (apiKey == null) {
			monitor.log("API key UID: " + iv.uid + " not found, ignoring");
			return;
		}

		RepositoryProvider.instance(IAPIKeyStore.class,
				ServerSideServiceProvider
						.getProvider(new SecurityContext("sid", apiKey.subject, List.of(), List.of(), apiKey.domainUid))
						.getContext())
				.delete(apiKey.uid);
	}

	// Pre BM 5.5, kafka ItemValue UID:
	// context.getSubject() + "-" + sid
	private void deleteApiKeyPre55(IServerTaskMonitor monitor, ItemValue<APIKey> iv) {
		Matcher matcher = REGEX_PRE_55_UID.matcher(iv.uid);
		if (!matcher.matches()) {
			monitor.log("Invalid API key itemvalue UID: " + iv.uid + ", ignoring");
			return;
		}

		String subject = matcher.group(1);
		String sid = matcher.group(2);

		APIKey apiKey = RepositoryProvider.instance(IAPIKeyStore.class,
				ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext()).get(sid);
		if (apiKey == null) {
			monitor.log("API key SID: " + sid + ", subject: " + subject + " not found, ignoring");
			return;
		}

		RepositoryProvider.instance(IAPIKeyStore.class,
				ServerSideServiceProvider
						.getProvider(new SecurityContext("sid", apiKey.subject, List.of(), List.of(), apiKey.domainUid))
						.getContext())
				.delete(apiKey.uid);
	}

	private void createApiKey(IServerTaskMonitor monitor, ItemValue<APIKey> iv) {
		monitor.log("Restore API key: " + iv);
		try {
			if (iv.value != null && iv.value.subject != null && iv.value.domainUid != null) {
				if (iv.value.uid == null || iv.value.uid.isBlank()) {
					iv.value.uid = UUID.randomUUID().toString();
				}

				BmContext ctx = ServerSideServiceProvider
						.getProvider(
								new SecurityContext("sid", iv.value.subject, List.of(), List.of(), iv.value.domainUid))
						.getContext();
				IAPIKeyStore store = RepositoryProvider.instance(IAPIKeyStore.class, ctx);
				store.create(iv.value);
				monitor.log("API key created for " + iv.value.displayName);
			}
		} catch (Exception e) {
			monitor.log("Error on retoring API key", e);
		}
	}
}
