/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm.user;

import java.util.Arrays;
import java.util.Base64;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Splitter;

import net.bluemind.config.Token;
import net.bluemind.core.api.Email;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainersFlatHierarchy;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.document.storage.IDocumentStore;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.eas.backend.bm.impl.CoreConnect;
import net.bluemind.eas.dto.user.MSUser;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailbox.api.Mailbox.Routing;
import net.bluemind.network.topology.Topology;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.User;

public class UserBackend extends CoreConnect {
	private static final Cache<String, MSUser> cache = Caffeine.newBuilder().recordStats()
			.expireAfterAccess(1, TimeUnit.MINUTES).build();

	public static class CacheRegistration implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register(UserBackend.class, cache);
		}
	}

	public static void purgeSession() {
		cache.invalidateAll();
	}

	public static List<String> purgeSession(String sid) {
		List<Entry<String, MSUser>> filter = cache.asMap().entrySet().stream()
				.filter(e -> e.getValue().getSid().equalsIgnoreCase(sid)).toList();
		filter.stream().map(e -> e.getKey()).forEach(cache::invalidate);
		return filter.stream().map(e -> e.getValue().getLoginAtDomain()).toList();
	}

	public MSUser getUser(String loginAtDomain, String sid) throws ActiveSyncException {
		String key = loginAtDomain + ":" + sid;
		MSUser ret = cache.getIfPresent(key);
		if (ret == null) {
			ret = getUserImpl(loginAtDomain, sid);
			cache.put(key, ret);
		} else {
			EasLogUser.logDebugAsUser(loginAtDomain, logger, "[{}] using cached user.", loginAtDomain);
		}
		return ret;
	}

	private MSUser getUserImpl(String loginAtDomain, String sid) throws ActiveSyncException {
		Iterator<String> latd = Splitter.on("@").split(loginAtDomain).iterator();
		@SuppressWarnings("unused")
		String login = latd.next();
		String domain = latd.next();
		try {
			String core = "http://" + Topology.get().core().value.address() + ":8090";
			// BM-8155
			IDomains domainsService = getService(core, Token.admin0(), IDomains.class);
			ItemValue<Domain> dom = domainsService.findByNameOrAliases(domain);

			ItemValue<User> user = getService(core, Token.admin0(), IUser.class, dom.uid).byEmail(loginAtDomain);
			return getUser(user, dom, sid);
		} catch (Exception e) {
			throw new ActiveSyncException(e);
		}
	}

	private String loadPhoto(Integer photoId, BackendSession bs) {
		// may not work, but you get the idea
		try {
			byte[] b = getService(bs, IDocumentStore.class).get(Integer.toString(photoId));
			return Base64.getEncoder().encodeToString(b);
		} catch (Exception e) {
			EasLogUser.logErrorAsUser(bs.getLoginAtDomain(), logger, "Fail to fetch photo {}", photoId);
		}
		return null;
	}

	public String getPictureBase64(BackendSession bs, int photoId) {
		return loadPhoto(photoId, bs);
	}

	public MSUser getUser(BackendSession bs, String userUid) {

		String core = "http://" + Topology.get().core().value.address() + ":8090";
		IUser userService = getService(core, Token.admin0(), IUser.class, bs.getUser().getDomain());
		IDomains domainsService = getService(core, Token.admin0(), IDomains.class);

		ItemValue<Domain> dom = domainsService.findByNameOrAliases(bs.getUser().getDomain());
		ItemValue<User> user = userService.getComplete(userUid);
		return getUser(user, dom, user.value.password);

	}

	private MSUser getUser(ItemValue<User> user, ItemValue<Domain> dom, String sid) {
		String core = "http://" + Topology.get().core().value.address() + ":8090";
		IUserSettings userSettingsService = getService(core, Token.admin0(), IUserSettings.class, dom.uid);
		Map<String, String> settings = userSettingsService.get(user.uid);
		IUser userService = getService(core, Token.admin0(), IUser.class, dom.uid);
		String lang = userService.getLocale(user.uid);
		String tz = settings.get("timezone");
		Set<String> emails = new HashSet<String>();
		String defaultEmail = user.value.defaultEmail().address;
		for (Email e : user.value.emails) {
			if (e.allAliases) {
				for (String alias : dom.value.aliases) {
					String email = e.address.split("@")[0] + "@" + alias;
					if (!defaultEmail.equals(email)) {
						emails.add(email);
					}
				}
			}
			emails.add(e.address);
		}
		MSUser ret = new MSUser(user.uid, user.displayName, user.value.login + "@" + dom.uid, sid, lang, tz,
				user.value.routing != Routing.none, user.value.defaultEmail().address, emails, user.value.dataLocation);
		return ret;
	}

	public boolean userHasRoleReadExtended(BackendSession bs, String containerUid) {
		IContainerManagement cmApi = getService(bs, IContainerManagement.class, containerUid);
		return cmApi.canAccessVerbs(Arrays.asList(Verb.ReadExtended.name())).can();
	}

	public Locale getUserLocale(MSUser user) {
		String core = "http://" + Topology.get().core().value.address() + ":8090";
		IUser userService = getService(core, Token.admin0(), IUser.class, user.getDomain());
		return Locale.of(userService.getLocale(user.getUid()));
	}

	public void touchOnMailbox(BackendSession bs) {
		MSUser user = bs.getUser();
		getService(bs, IContainersFlatHierarchy.class, user.getDomain(), user.getUid())
				.touch(IMailboxAclUids.uidForMailbox(user.getUid()));
	}
}
