/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm.mail;

import java.io.InputStream;
import java.lang.ref.Reference;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.MessageServiceFactory;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.message.MessageImpl;
import org.apache.james.mime4j.parser.MimeStreamParser;
import org.asynchttpclient.AsyncCompletionHandler;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.HttpResponseBodyPart;
import org.asynchttpclient.Response;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxItems;
import net.bluemind.backend.mail.api.ImapItemIdentifier;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxItem;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.FlagUpdate;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailApiHeaders;
import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.common.io.FileBackedOutputStream;
import net.bluemind.core.api.date.BmDateTime;
import net.bluemind.core.api.date.BmDateTimeWrapper;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerHierarchyNode;
import net.bluemind.core.container.api.ContainerSubscriptionModel;
import net.bluemind.core.container.api.IContainersFlatHierarchy;
import net.bluemind.core.container.api.IOwnerSubscriptions;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.eas.backend.BufferByteSource;
import net.bluemind.eas.backend.Changes;
import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.MSAttachementData;
import net.bluemind.eas.backend.MSEmail;
import net.bluemind.eas.backend.MailFolder;
import net.bluemind.eas.backend.SendMailData;
import net.bluemind.eas.backend.SendMailData.Mode;
import net.bluemind.eas.backend.bm.ContentsExporter.FilterTypeStatus;
import net.bluemind.eas.backend.bm.impl.CoreConnect;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.ContentImportEntityForChange;
import net.bluemind.eas.backend.importer.ContentImportEntityForDeletion;
import net.bluemind.eas.backend.importer.ContentImportEntityForMove;
import net.bluemind.eas.dto.base.AirSyncBaseResponse;
import net.bluemind.eas.dto.base.AppData;
import net.bluemind.eas.dto.base.BodyOptions;
import net.bluemind.eas.dto.base.ChangeType;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.base.DisposableByteSource;
import net.bluemind.eas.dto.base.LazyLoaded;
import net.bluemind.eas.dto.email.EmailResponse;
import net.bluemind.eas.dto.find.FindRequest;
import net.bluemind.eas.dto.find.FindResponse;
import net.bluemind.eas.dto.moveitems.MoveItemsResponse;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.dto.sync.CollectionSyncRequest;
import net.bluemind.eas.dto.sync.CollectionSyncRequest.Options.ConflicResolution;
import net.bluemind.eas.dto.sync.FilterType;
import net.bluemind.eas.dto.sync.SyncState;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.dto.user.MSUser;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.exception.CollectionNotFoundException;
import net.bluemind.eas.exception.NotAllowedException;
import net.bluemind.eas.exception.ObjectNotFoundException;
import net.bluemind.eas.exception.ServerErrorException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.imip.parser.IMIPInfos;
import net.bluemind.imip.parser.IMIPParserFactory;
import net.bluemind.mime4j.common.IMailRewriter;
import net.bluemind.mime4j.common.IRenderableMessage;
import net.bluemind.mime4j.common.Mime4JHelper;
import net.bluemind.mime4j.common.Mime4JHelper.HashedBuffer;
import net.bluemind.mime4j.common.RewriteMode;
import net.bluemind.mime4j.common.RewriterBuilder;
import net.bluemind.proxy.support.AHCWithProxy;

public class MailBackend extends CoreConnect {

	private final EmailManager emailManager;
	private final ISyncStorage storage;

	public MailBackend(ISyncStorage storage) {
		emailManager = EmailManager.getInstance();
		this.storage = storage;
	}

	public Changes getContentChanges(CollectionIdContext collectionIdContext, SyncState state,
			CollectionSyncRequest.Options options, FilterTypeStatus filterTypeStatus) throws ActiveSyncException {

		if (!collectionIdContext.backendSession().getUser().hasMailbox()) {
			EasLogUser.logInfoAsUser(collectionIdContext.getUserLogin(), logger,
					"MailRouting == NONE for user {}. Return no changes.", collectionIdContext.getUserLogin());
			return new Changes();
		}

		// Ensures window frame is respected when FilterType is active
		Optional<Instant> filteredDate = (options.filterType != null && options.filterType != FilterType.ALL_ITEMS)
				? Optional.of(options.filterType.filterDate())
				: Optional.empty();

		MailFolder folder = storage.getMailFolder(collectionIdContext);
		boolean isDraftFolder = "Drafts".equals(folder.fullName);

		IMailboxItems service = getMailboxItemsService(collectionIdContext.backendSession(), folder.uid);

		ContainerChangeset<ItemVersion> changeset = service.filteredChangesetById(state.version,
				ItemFlagFilter.create().mustNot(ItemFlag.Deleted));
		Changes changes = new Changes();
		changes.version = changeset.version;

		List<Long> created = new ArrayList<>();
		List<Long> softDeleted = new ArrayList<>();

		if (state.version > 0 && filterTypeStatus.hasFilterTypeChanged()) {
			EasLogUser.logDebugAsUser(collectionIdContext.getUserLogin(), logger, "FilterType changed: {} -> {}",
					filterTypeStatus.previousFilterTypeDate(), filterTypeStatus.currentFilterTypeDate());
			BmDateTime currentDate = (options.filterType == null)
					? BmDateTimeWrapper.fromTimestamp(Instant.EPOCH.toEpochMilli())
					: BmDateTimeWrapper.fromTimestamp(options.filterType.filterDate().toEpochMilli());

			Set<Long> currentItemIds = new HashSet<>(service.listItemIdsAfter(currentDate));

			BmDateTime previousDate = BmDateTimeWrapper
					.fromTimestamp(filterTypeStatus.previousFilterTypeDate().toEpochMilli());

			Set<Long> previousItemIds = new HashSet<>(service.listItemIdsAfter(previousDate));

			// items to create
			Set<Long> itemsToCreate = Sets.difference(currentItemIds, previousItemIds);
			created.addAll(itemsToCreate);

			// items to softDelete
			SetView<Long> itemsToSoftDelete = Sets.difference(previousItemIds, currentItemIds);
			softDeleted.addAll(itemsToSoftDelete);
		}
		created.addAll(changeset.created.stream().map(i -> i.id).toList());

		manageCreatedItems(collectionIdContext, filteredDate, service, created, changes);

		manageSoftDeletedItems(collectionIdContext, service, softDeleted, changes);

		changeSetUpdatedItems(collectionIdContext, options.bodyOptions, folder, isDraftFolder, service,
				changeset.updated, changes);

		changeSetDeletedItems(collectionIdContext, changeset.deleted, changes);

		return changes;

	}

	private void changeSetDeletedItems(CollectionIdContext collectionIdContext, List<ItemVersion> deleted,
			Changes changes) {
		deleted.forEach(itemVersion -> {
			ItemChangeReference ic = new ItemChangeReference(ItemDataType.EMAIL);
			ic.setServerId(CollectionItem.of(collectionIdContext.collectionId(), itemVersion.id));
			ic.setChangeType(ChangeType.DELETE);
			changes.items.add(ic);
		});
	}

	private void changeSetUpdatedItems(CollectionIdContext collectionIdContext, BodyOptions bodyOptions,
			MailFolder folder, boolean isDraftFolder, IMailboxItems service, List<ItemVersion> updated,
			Changes changes) {
		List<List<ItemVersion>> updatedParts = Lists.partition(updated, 250);
		for (List<ItemVersion> slice : updatedParts) {
			List<ItemValue<MailboxItem>> items = service.multipleGetById(slice.stream().map(v -> v.id).toList());
			items.forEach(item -> {
				boolean isDraft = item.value.flags.contains(MailboxItemFlag.System.Draft.value()) || isDraftFolder;
				if (item != null) {
					ItemChangeReference ic = new ItemChangeReference(ItemDataType.EMAIL);
					ic.setServerId(CollectionItem.of(collectionIdContext.collectionId(), item.internalId));
					ic.setChangeType(ChangeType.CHANGE);

					if (isDraft) {
						ic.setData(toAppData(collectionIdContext.backendSession(), bodyOptions, folder,
								ic.getServerId().itemId));
					} else {
						ic.setData(AppData.of(FlagsChange.asEmailResponse(item.value), LazyLoaded.NOOP));
					}
					changes.items.add(ic);
				}
			});
		}
	}

	private void manageSoftDeletedItems(CollectionIdContext collectionIdContext, IMailboxItems service,
			List<Long> softDeleted, Changes changes) {
		List<List<Long>> softDeleteddParts = Lists.partition(softDeleted, 250);
		for (List<Long> slice : softDeleteddParts) {
			List<ItemValue<MailboxItem>> items = service.multipleGetById(slice);
			items.forEach(item -> {
				if (item != null) {
					ItemChangeReference ic = new ItemChangeReference(ItemDataType.EMAIL);
					ic.setServerId(CollectionItem.of(collectionIdContext.collectionId(), item.internalId));
					ic.setChangeType(ChangeType.SOFTDELETE);
					changes.items.add(ic);
				}
			});
		}
	}

	private void manageCreatedItems(CollectionIdContext collectionIdContext, Optional<Instant> filteredDate,
			IMailboxItems service, List<Long> created, Changes changes) {
		if (!filteredDate.isPresent()) {
			created.forEach(id -> {
				ItemChangeReference ic = new ItemChangeReference(ItemDataType.EMAIL);
				ic.setServerId(CollectionItem.of(collectionIdContext.collectionId(), id));
				ic.setChangeType(ChangeType.ADD);
				changes.items.add(ic);
			});
		} else {
			Instant deliveredAfter = filteredDate.get();
			List<List<Long>> createdParts = Lists.partition(created, 250);
			AtomicBoolean stopLoading = new AtomicBoolean(false);
			AtomicInteger addedToSync = new AtomicInteger(0);
			for (List<Long> slice : createdParts) {
				if (stopLoading.get()) {
					break;
				}
				service.multipleGetById(slice).stream().filter(item -> item != null && item.value != null)
						.forEach(item -> {
							deliveredAfter.isBefore(Instant.ofEpochMilli(item.value.body.date.getTime()));
							if (deliveredAfter.isBefore(Instant.ofEpochMilli(item.value.body.date.getTime()))) {
								ItemChangeReference ic = new ItemChangeReference(ItemDataType.EMAIL);
								ic.setServerId(CollectionItem.of(collectionIdContext.collectionId(), item.internalId));
								ic.setChangeType(ChangeType.ADD);
								changes.items.add(ic);
								addedToSync.incrementAndGet();
							} else {
								EasLogUser.logDebugAsUser(collectionIdContext.getUserLogin(), logger,
										"[{}] Stop loading at email {} ({} is before {}), {} / {}", //
										collectionIdContext.getUserLogin(), item.value, item.value.body.date,
										deliveredAfter, addedToSync, created.size());
								// stop loading as the changeset is sorted by
								// delivery date
								stopLoading.set(true);
								return;
							}
						});
			}
		}
	}

	public void delete(ContentImportEntityForDeletion contentEntity) throws CollectionNotFoundException {
		if (contentEntity.serverIds != null && !contentEntity.serverIds.isEmpty()) {
			HashMap<MailFolder, List<Long>> items = getMailFolders(contentEntity);

			MSUser msUser = contentEntity.backendSession.getUser();
			for (Entry<MailFolder, List<Long>> entry : items.entrySet()) {
				MailFolder folder = entry.getKey();
				if (contentEntity.moveToTrash) {
					moveToTrash(contentEntity, items, msUser, folder);
				} else {
					deleteMail(contentEntity, msUser.getUid(), entry.getValue(), folder.uid);
				}
			}
		}
	}

	private void deleteMail(ContentImportEntityForDeletion contentEntity, String userUid, List<Long> entries,
			String folderUid) {
		IMailboxItems service = getMailboxItemsService(contentEntity.backendSession, folderUid);
		entries.forEach(id -> {
			try {
				EasLogUser.logInfoAsUser(contentEntity.backendSession.getLoginAtDomain(), logger, "[{}] Delete mail {}",
						userUid, id);
				service.deleteById(id);
			} catch (ServerFault sf) {
				if (sf.getCode() != ErrorCode.TIMEOUT) {
					throw sf;
				}
			}
		});
	}

	private void moveToTrash(ContentImportEntityForDeletion contentEntity, HashMap<MailFolder, List<Long>> items,
			MSUser msUser, MailFolder folder) throws CollectionNotFoundException {
		String mailboxUid = contentEntity.backendSession.getUser().getUid();
		if (folder.collectionId.getSubscriptionId().isPresent()) {
			IOwnerSubscriptions subscriptionsService = getService(contentEntity.backendSession,
					IOwnerSubscriptions.class, msUser.getDomain(), msUser.getUid());
			ItemValue<ContainerSubscriptionModel> sub = subscriptionsService
					.getCompleteById(folder.collectionId.getSubscriptionId().get());
			mailboxUid = sub.value.owner;
		}
		IMailboxFolders service = getMailboxFoldersServiceByCollection(
				new CollectionIdContext(contentEntity.backendSession, folder.collectionId));
		ItemValue<MailboxFolder> source = service.getComplete(folder.uid);
		HierarchyNode sourceHierarchyNode = storage.getHierarchyNode(contentEntity.backendSession.getUniqueIdentifier(),
				msUser.getDomain(), mailboxUid, ContainerHierarchyNode.uidFor(IMailReplicaUids.mboxRecords(source.uid),
						"mailbox_records", msUser.getDomain()));

		CyrusPartition part = CyrusPartition.forServerAndDomain(msUser.getDataLocation(), msUser.getDomain());
		IMailboxFolders mboxFolders = getService(contentEntity.backendSession, IMailboxFolders.class, part.name,
				"user." + msUser.getUid().replace('.', '^'));
		ItemValue<MailboxFolder> trash = mboxFolders.byName("Trash");
		HierarchyNode trashHierarchyNode = storage.getHierarchyNode(contentEntity.backendSession.getUniqueIdentifier(),
				msUser.getDomain(), msUser.getUid(), ContainerHierarchyNode
						.uidFor(IMailReplicaUids.mboxRecords(trash.uid), "mailbox_records", msUser.getDomain()));

		emailManager.moveItems(contentEntity.backendSession, sourceHierarchyNode, trashHierarchyNode,
				items.get(folder).stream().map(i -> (long) i).toList());
	}

	private HashMap<MailFolder, List<Long>> getMailFolders(ContentImportEntityForDeletion contentEntity)
			throws CollectionNotFoundException {
		HashMap<String, MailFolder> collections = new HashMap<>();
		HashMap<MailFolder, List<Long>> items = new HashMap<>();
		for (CollectionItem serverId : contentEntity.serverIds) {
			String collectionId = serverId.collectionId.getValue();
			if (!collections.containsKey(collectionId)) {
				MailFolder folder = storage
						.getMailFolder(new CollectionIdContext(contentEntity.backendSession, serverId.collectionId));
				collections.put(collectionId, folder);
				items.put(folder, new ArrayList<>());
			}

			items.get(collections.get(collectionId)).add(serverId.itemId);
		}
		return items;
	}

	public void sendDraft(ContentImportEntityForChange contentEntity) throws ActiveSyncException {

		MSEmail email = (MSEmail) contentEntity.data;
		CollectionItem ci = CollectionItem.of(contentEntity.serverId.get());

		MailFolder folder = storage
				.getMailFolder(new CollectionIdContext(contentEntity.backendSession, ci.collectionId));
		IMailboxItems service = getMailboxItemsService(contentEntity.backendSession, folder.uid);

		ItemValue<MailboxItem> draft = service.getCompleteById(ci.itemId);
		MessageImpl message = email.getMessage();
		BodyMailLoader.mergeDraft(draft, message);

		SendMailData mailData = new SendMailData();
		mailData.backendSession = contentEntity.backendSession;
		HashedBuffer hashedBuffer = null;
		try {
			if (email.getMimeContent() != null) {
				mailData.mailContent = email.getMimeContent();
			} else {
				try {
					hashedBuffer = Mime4JHelper.mmapedEML(message);
					mailData.mailContent = BufferByteSource.of(hashedBuffer.nettyBuffer());
				} catch (Exception e) {
					throw new ActiveSyncException(e.getMessage());
				}
			}
			mailData.saveInSent = true;
			mailData.mode = Mode.Send;

			sendEmail(mailData);
		} finally {
			Reference.reachabilityFence(hashedBuffer);
		}
	}

	public CollectionItem store(ContentImportEntityForChange contentEntity) throws ActiveSyncException {
		MailFolder folder = storage
				.getMailFolder(new CollectionIdContext(contentEntity.backendSession, contentEntity.collectionId));
		IMailboxItems service = getMailboxItemsService(contentEntity.backendSession, folder.uid);
		MSEmail email = (MSEmail) contentEntity.data;
		if (contentEntity.serverId.isPresent()) {
			return updateDraft(contentEntity, folder, service, email);
		} else {
			return addDraft(contentEntity.collectionId, service, email);
		}
	}

	private CollectionItem addDraft(CollectionId collectionId, IMailboxItems service, MSEmail email)
			throws ActiveSyncException {
		MessageBody messageBody;
		if (email.getMimeContent() != null) {
			messageBody = BodyMailLoader.uploadCompleteEml(service, email.getMimeContent());
		} else {
			try {
				HashedBuffer hashedBuffer = Mime4JHelper.mmapedEML(email.getMessage());
				try {
					messageBody = BodyMailLoader.uploadCompleteEml(service,
							BufferByteSource.of(hashedBuffer.nettyBuffer()));
				} finally {
					Reference.reachabilityFence(hashedBuffer);
				}
			} catch (Exception e) {
				throw new ActiveSyncException(e.getMessage());
			}
		}

		MailboxItem mailboxItem = new MailboxItem();
		mailboxItem.body = messageBody;
		mailboxItem.flags = Arrays.asList(MailboxItemFlag.System.Draft.value(), MailboxItemFlag.System.Seen.value());
		ImapItemIdentifier created = service.create(mailboxItem);
		return CollectionItem.of(collectionId, created.id);
	}

	private CollectionItem updateDraft(ContentImportEntityForChange contentEntity, MailFolder folder,
			IMailboxItems service, MSEmail email) throws ActiveSyncException {
		CollectionItem ci = CollectionItem.of(contentEntity.serverId.get());
		if ("Drafts".equals(folder.fullName)) {
			updateBody(contentEntity, service, email, ci);
		} else {
			try {
				validateFlag(email.isRead(), MailboxItemFlag.System.Seen.value(), service, ci.itemId);
				validateFlag(email.isStarred(), MailboxItemFlag.System.Flagged.value(), service, ci.itemId);
			} catch (ServerFault e) {
				if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
					throw new NotAllowedException(e);
				} else {
					throw e;
				}
			}
		}
		return ci;
	}

	public void updateBody(ContentImportEntityForChange contentEntity, IMailboxItems mailboxItemService, MSEmail email,
			CollectionItem ci) throws ActiveSyncException {
		ItemValue<MailboxItem> msgFromServer = null;
		String user = contentEntity.user;
		try {
			msgFromServer = mailboxItemService.getForUpdate(ci.itemId);
		} catch (Exception e) {
			EasLogUser.logDebugAsUser(user, logger, "Fail to find MailboxItem {}", ci.itemId);
			throw new ObjectNotFoundException("Fail to find MailboxItem " + ci.itemId);
		}

		DraftSynchronization draftSync = null;
		MessageBody msgBodyFromDevice = null;
		if (contentEntity.conflictPolicy == ConflicResolution.SERVER_WINS
				&& msgFromServer.version > contentEntity.syncState.version) {
			String msg = String.format(
					"Both server (version '%d') and client (version '%d') changes. Conflict resolution is SERVER_WINS for message %s::%s",
					msgFromServer.version, contentEntity.syncState.version, contentEntity.collectionId,
					contentEntity.serverId.get());
			throw new ActiveSyncException(msg);
		}

		if (email.getMimeContent() != null) {
			msgBodyFromDevice = BodyMailLoader.uploadCompleteEml(mailboxItemService, email.getMimeContent());
			draftSync = new DraftSynchronization(user, mailboxItemService, msgFromServer, email, ci, msgBodyFromDevice);
		} else {
			try {
				BodyMailLoader.mergeDraft(msgFromServer, email.getMessage());
				draftSync = new DraftSynchronization(user, mailboxItemService, msgFromServer, email, ci);
				draftSync.injectChanges();
			} catch (Exception e) {
				EasLogUser.logExceptionAsUser(user, e, logger);
				return;
			}
		}

		if (draftSync.isUpdateDraft()) {
			updateRefreshDateHeader(draftSync.getMsgFromServer());
			draftSync.getMsgFromServer().flags = Arrays.asList(MailboxItemFlag.System.Draft.value(),
					MailboxItemFlag.System.Seen.value());
			try {
				mailboxItemService.updateById(ci.itemId, draftSync.getMsgFromServer());
				EasLogUser.logInfoAsUser(user, logger, "Draft [{}] '{}' refreshed and updated on server.",
						ci.toString(), draftSync.getMsgFromServer().body.subject);
			} finally {
				draftSync.getAttachmentPartsToRemoveAfter().forEach(a -> mailboxItemService.removePart(a.address));
			}
		}
	}

	private static void updateRefreshDateHeader(MailboxItem draft) {
		draft.body.headers.removeIf(header -> header.name.equals(MailApiHeaders.X_BM_DRAFT_REFRESH_DATE));
		draft.body.headers.add(
				MessageBody.Header.create(MailApiHeaders.X_BM_DRAFT_REFRESH_DATE, "" + System.currentTimeMillis()));
	}

	private static void validateFlag(Boolean property, MailboxItemFlag flag, IMailboxItems service, long itemId) {
		if (property != null) {
			if (property.booleanValue()) {
				service.addFlag(FlagUpdate.of(itemId, flag));
			} else {
				service.deleteFlag(FlagUpdate.of(itemId, flag));
			}
		}

	}

	public List<MoveItemsResponse.Response> move(ContentImportEntityForMove contentEntity) {
		return emailManager.moveItems(contentEntity.backendSession, contentEntity.srcFolder, contentEntity.dstFolder,
				contentEntity.items.stream().map(v -> v.itemId).collect(Collectors.toList()));
	}

	/**
	 * @param mail
	 */
	public void sendEmail(SendMailData mail) throws ActiveSyncException {
		BackendSession bs = mail.backendSession;

		if (!bs.getUser().hasMailbox()) {
			EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger,
					"MailRouting == NONE for user {}. Do not try to send mail", bs.getLoginAtDomain());
			return;
		}

		try {
			Message m = MessageServiceFactory.newInstance().newMessageBuilder()
					.parseMessage(mail.mailContent.openBufferedStream());
			IMIPInfos infos = IMIPParserFactory.create().parse(m);
			boolean isSmime = MailApiHeaders.MIME_TYPE_SMIME_SIGNED.equals(m.getMimeType());

			if (infos == null) {
				IMailRewriter rewriter = Mime4JHelper.untouched(getUserEmail(bs));
				send(bs, mail.mailContent, rewriter, mail.saveInSent, isSmime);
			} else {
				ICalendar cs = getService(bs, ICalendar.class,
						ICalendarUids.defaultUserCalendar(bs.getUser().getUid()));
				new MailImipManager(bs, infos, cs).processImipInfos();
			}
		} catch (Exception e) {
			throw new ServerErrorException(e);
		}
	}

	/**
	 * @param backendSession
	 * @param mailContent
	 * @param saveInSent
	 * @param collectionId
	 * @param serverId
	 * @throws Exception
	 */
	public void replyToEmail(CollectionIdContext collectionIdContext, ByteSource mailContent, Boolean saveInSent,
			String serverId, boolean includePrevious) throws Exception {
		MailFolder folder = storage.getMailFolder(collectionIdContext);
		long id = CollectionItem.of(serverId).itemId;

		IMailRewriter rewriter = Mime4JHelper.untouched(getUserEmail(collectionIdContext.backendSession()));
		if (includePrevious) {
			try (InputStream is = emailManager.fetchMimeStream(collectionIdContext.backendSession(), folder, id)) {
				if (is != null) {
					RewriterBuilder rb = new RewriterBuilder();
					rb.setMode(RewriteMode.REPLY);
					rb.setKeepAttachments(false);
					rb.setIncludedContent(is);
					rb.setFrom(getUserEmail(collectionIdContext.backendSession()));
					rewriter = rb.build();
				}
			}
		}
		IMailboxItems service = getMailboxItemsService(collectionIdContext.backendSession(), folder.uid);
		service.addFlag(FlagUpdate.of(id, MailboxItemFlag.System.Answered.value()));

		send(collectionIdContext.backendSession(), mailContent, rewriter, saveInSent, false);
	}

	public void forwardEmail(CollectionIdContext collectionIdContext, ByteSource mailContent, Boolean saveInSent,
			String serverId, boolean includePrevious) throws Exception {
		MailFolder folder = storage.getMailFolder(collectionIdContext);
		long id = CollectionItem.of(serverId).itemId;

		IMailRewriter rewriter = Mime4JHelper.untouched(getUserEmail(collectionIdContext.backendSession()));
		if (includePrevious) {
			try (InputStream is = emailManager.fetchMimeStream(collectionIdContext.backendSession(), folder, id)) {
				if (is != null) {
					RewriterBuilder rb = new RewriterBuilder();
					rb.setMode(RewriteMode.FORWARD_INLINE);
					rb.setKeepAttachments(true);
					rb.setIncludedContent(is);
					rb.setFrom(getUserEmail(collectionIdContext.backendSession()));
					rewriter = rb.build();
				}
			}
		}

		IMailboxItems service = getMailboxItemsService(collectionIdContext.backendSession(), folder.uid);
		service.addFlag(FlagUpdate.of(id, new MailboxItemFlag("$Forwarded")));

		send(collectionIdContext.backendSession(), mailContent, rewriter, saveInSent, false);
	}

	private Mailbox getUserEmail(BackendSession bs) {
		MSUser u = bs.getUser();
		String from = u.getDefaultEmail();
		String dn = u.getDisplayName();
		String[] split = from.split("@");
		return new Mailbox(dn, split[0], split[1]);
	}

	private void send(BackendSession bs, ByteSource mailContent, IMailRewriter handler, Boolean saveInSent,
			boolean isSmime) throws Exception {
		IRenderableMessage render = handler;
		MimeStreamParser parser = Mime4JHelper.parser();
		parser.setContentHandler(handler);
		parser.parse(mailContent.openBufferedStream());
		if (isSmime) {
			render = new IRenderableMessage() {

				@Override
				public InputStream renderAsMimeStream() throws Exception {
					return mailContent.openBufferedStream();
				}

				@Override
				public <T> T renderAs(Class<T> klass) throws Exception {
					return handler.renderAs(klass);
				}
			};
		}
		emailManager.sendEmail(bs, render, saveInSent);
	}

	/**
	 * @param bs
	 * @param attachmentId
	 * @return
	 * @throws ObjectNotFoundException
	 */
	public MSAttachementData getAttachment(BackendSession bs, String attachmentId) throws ActiveSyncException {
		if (attachmentId != null && !attachmentId.isEmpty()) {
			Map<String, String> parsedAttId = AttachmentHelper.parseAttachmentId(attachmentId);
			try {

				String type = parsedAttId.get(AttachmentHelper.TYPE);
				if (AttachmentHelper.BM_FILEHOSTING.equals(type)) {
					String url = parsedAttId.get(AttachmentHelper.URL);
					String contentType = parsedAttId.get(AttachmentHelper.CONTENT_TYPE);

					try (FileBackedOutputStream fbos = new FileBackedOutputStream(32000, "bm-eas-getattachment");
							AsyncHttpClient ahc = AHCWithProxy.build(storage.getSystemConf())) {
						return ahc.prepareGet(url).execute(new AsyncCompletionHandler<MSAttachementData>() {

							@Override
							public State onBodyPartReceived(HttpResponseBodyPart bodyPart) throws Exception {
								fbos.write(bodyPart.getBodyPartBytes());
								return State.CONTINUE;
							}

							@Override
							public MSAttachementData onCompleted(Response response) throws Exception {
								return new MSAttachementData(contentType, DisposableByteSource.wrap(fbos));
							}
						}).get(20, TimeUnit.SECONDS);
					}
				}

				String collectionId = parsedAttId.get(AttachmentHelper.COLLECTION_ID);
				String messageId = parsedAttId.get(AttachmentHelper.MESSAGE_ID);
				String mimePartAddress = parsedAttId.get(AttachmentHelper.MIME_PART_ADDRESS);
				String contentType = parsedAttId.get(AttachmentHelper.CONTENT_TYPE);
				String contentTransferEncoding = parsedAttId.get(AttachmentHelper.CONTENT_TRANSFER_ENCODING);
				EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger,
						"attachmentId: [colId:{}] [emailUid:{}] [partAddress:{}] [contentType:{}] [transferEncoding:{}]",
						collectionId, messageId, mimePartAddress, contentType, contentTransferEncoding);

				MailFolder folder = storage.getMailFolder(new CollectionIdContext(bs, CollectionId.of(collectionId)));

				InputStream is = emailManager.fetchAttachment(bs, folder, Long.parseLong(messageId), mimePartAddress,
						contentTransferEncoding);
				byte[] bytes = ByteStreams.toByteArray(is);
				is.close();

				return new MSAttachementData(contentType, DisposableByteSource.wrap(bytes));
			} catch (Exception e) {
				throw new ActiveSyncException(e);
			}
		}
		throw new ObjectNotFoundException(String.format("Failed to fetch attachment %s", attachmentId));
	}

	public void purgeFolder(CollectionIdContext collectionIdContext, boolean deleteSubFolder)
			throws CollectionNotFoundException, NotAllowedException {
		MailFolder folder = storage.getMailFolder(collectionIdContext);

		if (!"Trash".equals(folder.fullName)) {
			throw new NotAllowedException("Only the Trash folder can be purged.");
		}
		emailManager.purgeFolder(collectionIdContext, folder, deleteSubFolder);
	}

	public AppData fetch(BackendSession bs, BodyOptions bodyParams, ItemChangeReference ic) throws ActiveSyncException {
		try {
			MailFolder folder = storage.getMailFolder(new CollectionIdContext(bs, ic.getServerId().collectionId));
			return toAppData(bs, bodyParams, folder, ic.getServerId().itemId);
		} catch (ActiveSyncException ase) {
			throw ase;
		} catch (Exception e) {
			throw new ActiveSyncException("Fail to fetch mail in collection " + ic.getServerId().collectionId, e);
		}
	}

	public Map<Long, AppData> fetchMultiple(CollectionIdContext collectionIdContext, BodyOptions bodyParams,
			List<Long> ids) throws ActiveSyncException {

		MailFolder folder = storage.getMailFolder(collectionIdContext);

		Map<Long, AppData> res = HashMap.newHashMap(ids.size());
		ids.stream().forEach(id -> {
			try {
				AppData data = toAppData(collectionIdContext.backendSession(), bodyParams, folder, id);
				res.put(id, data);
			} catch (Exception e) {
				EasLogUser.logWarnAsUser(collectionIdContext.getUserLogin(), logger,
						"Fail to convert email {}, folder {}. Skip it.", id, folder);
			}
		});

		return res;
	}

	private AppData toAppData(BackendSession bs, BodyOptions bodyParams, MailFolder folder, Long id) {
		EmailResponse er = EmailManager.getInstance().loadStructure(bs, folder, id);
		LazyLoaded<BodyOptions, AirSyncBaseResponse> bodyProvider = BodyLoaderFactory.from(bs, folder, id, bodyParams);
		return AppData.of(er, bodyProvider);
	}

	public FindResponse.Response find(BackendSession bs, FindRequest query) throws CollectionNotFoundException {
		FindMail findMail = new FindMail(bs, query);

		if (query.executeSearch.mailBoxSearchCriterion != null) {
			IMailboxFolders mailboxFolderService = getMailboxFoldersService(bs);
			IContainersFlatHierarchy flatH = getService(bs, IContainersFlatHierarchy.class, bs.getUser().getDomain(),
					bs.getUser().getUid());
			return findMail.search(storage, mailboxFolderService, flatH);
		}

		return FindMail.invalidRequest();
	}

}
