/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm.contacts;

import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Kind;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.eas.backend.Changes;
import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.MSContact;
import net.bluemind.eas.backend.bm.compat.OldFormats;
import net.bluemind.eas.backend.bm.impl.CoreConnect;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.ContentImportEntityForChange;
import net.bluemind.eas.backend.importer.ContentImportEntityForDeletion;
import net.bluemind.eas.dto.base.AirSyncBaseResponse;
import net.bluemind.eas.dto.base.AppData;
import net.bluemind.eas.dto.base.BodyType;
import net.bluemind.eas.dto.base.ChangeType;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.base.DisposableByteSource;
import net.bluemind.eas.dto.base.LazyLoaded;
import net.bluemind.eas.dto.contact.ContactResponse;
import net.bluemind.eas.dto.sync.CollectionSyncRequest.Options.ConflicResolution;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;

/**
 * Contacts backend implementation
 *
 *
 */
public class ContactsBackend extends CoreConnect {

	private final ISyncStorage storage;

	public ContactsBackend(ISyncStorage storage) {
		this.storage = storage;
	}

	public Changes getContentChanges(CollectionIdContext collectionIdContext, long version) {
		Changes changes = new Changes();

		try {
			HierarchyNode folder = storage.getHierarchyNode(collectionIdContext);
			IAddressBook service = getAddressbookService(collectionIdContext.backendSession(), folder.containerUid);

			ContainerChangeset<Long> changeset = service.changesetById(version);
			EasLogUser.logDebugAsUser(collectionIdContext.getUserLogin(), logger,
					"[{}][{}] get contacts changes. created: {}, updated: {}, deleted: {}, folder: {}, version: {}",
					collectionIdContext.getUserLogin(), collectionIdContext.backendSession().getDevId(),
					changeset.created.size(), changeset.updated.size(), changeset.deleted.size(), folder.containerUid,
					version);

			changes.version = changeset.version;

			for (long id : changeset.created) {
				changes.items.add(
						getItemChange(collectionIdContext.collectionId(), id, ItemDataType.CONTACTS, ChangeType.ADD));
			}

			for (long id : changeset.updated) {
				changes.items.add(getItemChange(collectionIdContext.collectionId(), id, ItemDataType.CONTACTS,
						ChangeType.CHANGE));
			}

			for (long id : changeset.deleted) {
				ItemChangeReference ic = getItemChange(collectionIdContext.collectionId(), id, ItemDataType.CONTACTS,
						ChangeType.DELETE);
				changes.items.add(ic);
			}

		} catch (ServerFault e) {
			if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
				EasLogUser.logWarnAsUser(collectionIdContext.getUserLogin(), logger, e.getMessage());
			} else {
				EasLogUser.logExceptionAsUser(collectionIdContext.getUserLogin(), e, logger);
			}
			changes.version = version;
		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(collectionIdContext.getUserLogin(), e, logger);
			// BM-7227
			// Something went wrong
			// Send current version number to prevent full sync
			changes.version = version;
		}

		return changes;
	}

	public CollectionItem store(ContentImportEntityForChange contentEntity) throws ActiveSyncException {
		CollectionItem ret = null;
		try {
			HierarchyNode folder = storage.getHierarchyNode(contentEntity.collectionIdContext);
			IAddressBook service = getAddressbookService(contentEntity.backendSession, folder.containerUid);
			String user = contentEntity.backendSession.getLoginAtDomain();

			MSContact d = (MSContact) contentEntity.data;

			ContactConverter contactConverter = new ContactConverter();
			VCard vcard = contactConverter.contact(d);
			Long id = null;
			String uid = null;
			if (contentEntity.serverId.isPresent()) {
				EasLogUser.logInfoAsUser(user, logger, "update in " + contentEntity.collectionId + " (contact: "
						+ d.getFirstName() + " " + d.getLastName() + ")");
				String serverId = contentEntity.serverId.get();
				id = getItemId(serverId);
				ItemValue<VCard> prevVersion = service.getCompleteById(id);
				if (prevVersion == null) {
					EasLogUser.logDebugAsUser(user, logger, "Fail to find VCard {}", id);
					return CollectionItem.of(contentEntity.collectionId, id);
				}

				uid = prevVersion.uid;

				if (contentEntity.conflictPolicy == ConflicResolution.SERVER_WINS
						&& prevVersion.version > contentEntity.syncState.version) {
					String msg = String.format(
							"Both server (version '%d') and client (version '%d') changes. Conflict resolution is SERVER_WINS for contact %s::%s",
							prevVersion.version, contentEntity.syncState.version, contentEntity.collectionId, serverId);
					throw new ActiveSyncException(msg);
				}

				if (prevVersion.value.identification.photo && (d.getPicture() == null || d.getPicture().isEmpty())) {
					service.deletePhoto(uid);
				}

				contactConverter.mergeEmails(vcard, prevVersion.value);

				service.updateById(id, vcard);
			} else {
				uid = UUID.randomUUID().toString();
				EasLogUser.logInfoAsUser(user, logger, "create in " + contentEntity.collectionId + " (contact: "
						+ d.getFirstName() + " " + d.getLastName() + ")");
				service.create(uid, vcard);

				ItemValue<VCard> created = service.getComplete(uid);
				id = created.internalId;
			}

			if (d.getPicture() != null && !d.getPicture().isEmpty()) {
				try {
					EasLogUser.logInfoAsUser(user, logger, "Setting picture for contact");
					service.setPhoto(uid, Base64.getDecoder().decode(d.getPicture()));
				} catch (Exception e) {
					EasLogUser.logErrorExceptionAsUser(user, e, logger, "Fail to set contact picture");
				}
			}
			ret = CollectionItem.of(contentEntity.collectionId, id);
		} catch (ServerFault e) {
			throw new ActiveSyncException(e);
		}

		return ret;

	}

	public void delete(ContentImportEntityForDeletion contentEntity) throws ActiveSyncException {
		if (contentEntity.serverIds != null) {
			try {

				for (CollectionItem serverId : contentEntity.serverIds) {
					HierarchyNode folder = storage.getHierarchyNode(
							new CollectionIdContext(contentEntity.backendSession, serverId.collectionId));
					IAddressBook service = getAddressbookService(contentEntity.backendSession, folder.containerUid);
					service.deleteById(serverId.itemId);
				}
			} catch (ServerFault e) {
				if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
					throw new ActiveSyncException(e);
				}
				EasLogUser.logExceptionAsUser(contentEntity.user, e, logger);
			}
		}
	}

	public AppData fetch(BackendSession bs, ItemChangeReference ic) throws ActiveSyncException {
		try {
			HierarchyNode folder = storage.getHierarchyNode(new CollectionIdContext(bs, ic.getServerId().collectionId));
			IAddressBook service = getAddressbookService(bs, folder.containerUid);

			ItemValue<VCard> vcard = service.getCompleteById(ic.getServerId().itemId);
			AppData ret = toAppData(service, vcard);

			return ret;
		} catch (Exception e) {
			throw new ActiveSyncException(e.getMessage(), e);
		}
	}

	public Map<Long, AppData> fetchMultiple(CollectionIdContext collectionIdContext, List<Long> ids)
			throws ActiveSyncException {
		HierarchyNode folder = storage.getHierarchyNode(collectionIdContext);
		IAddressBook service = getAddressbookService(collectionIdContext.backendSession(), folder.containerUid);

		List<ItemValue<VCard>> vcards = service.multipleGetById(ids);
		Map<Long, AppData> res = new HashMap<>(ids.size());

		vcards.stream().filter(v -> v.value.kind == Kind.individual).forEach(vcard -> {
			try {
				AppData data = toAppData(service, vcard);
				res.put(vcard.internalId, data);
			} catch (Exception e) {
				EasLogUser.logErrorExceptionAsUser(collectionIdContext.getUserLogin(), e, logger,
						"Fail to convert vcard {}", vcard.uid);
			}
		});

		return res;
	}

	private AppData toAppData(IAddressBook service, ItemValue<VCard> vcard) {
		MSContact msContact = new ContactConverter().convert(service, vcard);

		ContactResponse cr = OldFormats.update(msContact);
		AppData data = AppData.of(cr);

		if (!msContact.getData().trim().isEmpty()) {
			final AirSyncBaseResponse airSyncBase = new AirSyncBaseResponse();
			airSyncBase.body = new AirSyncBaseResponse.Body();
			airSyncBase.body.type = BodyType.PlainText;
			airSyncBase.body.data = DisposableByteSource.wrap(msContact.getData().trim());
			airSyncBase.body.estimatedDataSize = (int) airSyncBase.body.data.size();
			data.body = LazyLoaded.loaded(airSyncBase);
		}
		return data;
	}

}
