/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.domain.service.property;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.system.api.ICredentials;

public class Obfuscate {
	private static final Set<String> properties = getPropertiesToObfuscate();

	private static final Set<String> getPropertiesToObfuscate() {
		return new RunnableExtensionLoader<ObfuscateRegistrator>()
				.loadExtensions("net.bluemind.core.domain.service", "property.obfuscate", "registrator", "impl")
				.stream().map(ObfuscateRegistrator::registerPropertyName).filter(Objects::nonNull).flatMap(Set::stream)
				.collect(Collectors.toSet());
	}

	public boolean isObfuscated(String propertyName) {
		return properties.stream().anyMatch(obfuscatedProperty -> this.needObfuscate(obfuscatedProperty, propertyName));
	}

	private boolean needObfuscate(String obfuscatedProperty, String propertyName) {
		if (obfuscatedProperty.startsWith("*")) {
			return propertyName.endsWith(obfuscatedProperty.substring(1));
		}

		return obfuscatedProperty.equals(propertyName);
	}

	public String obfuscatedValue(String propertyName, String propertyValue) {
		if (isObfuscated(propertyName)) {
			return ICredentials.OBFUSCATED_CREDENTIAL_VALUE;
		}

		return propertyValue;
	}

	public String plainValue(Map<String, String> oldProperties, String propertyName, String newValue) {
		if (!isObfuscated(propertyName) || !oldProperties.containsKey(propertyName)) {
			return newValue;
		}

		if (newValue != null && (newValue.isEmpty() || newValue.equals(ICredentials.OBFUSCATED_CREDENTIAL_VALUE))) {
			return oldProperties.get(propertyName);
		}

		return newValue;
	}
}
