package net.bluemind.delivery.lmtp.common;

import java.io.IOException;

import org.apache.james.mime4j.dom.Message;

import com.google.common.hash.Hashing;
import com.google.common.hash.HashingOutputStream;

import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.memory.pool.api.ChunkFiler.NamedChunk;
import net.bluemind.mime4j.common.Mime4JHelper;
import net.bluemind.size.helper.MaxMessageSize;

public class FreezableDeliveryContent {

	public record SerializedMessage(NamedChunk buffer, String guid) {
		public long size() {
			return buffer.chunk().getSize();
		}
	}

	private final DeliveryContent content;
	private final long size;
	private SerializedMessage serializedMessage;

	private FreezableDeliveryContent(DeliveryContent content, long size) {
		this.content = content;
		this.size = size;
	}

	private FreezableDeliveryContent(DeliveryContent content, SerializedMessage serializedMessage) {
		this(content, serializedMessage.size());
		this.serializedMessage = serializedMessage;
	}

	public DeliveryContent content() {
		return content.withSize(size);
	}

	public long size() {
		return size();
	}

	public SerializedMessage serializedMessage() {
		return serializedMessage;
	}

	public boolean isEmpty() {
		return content.isEmpty();
	}

	public boolean isFrozen() {
		return serializedMessage != null;
	}

	public static FreezableDeliveryContent create(DeliveryContent content, long size) throws IOException {
		return new FreezableDeliveryContent(content, size);
	}

	public static FreezableDeliveryContent freeze(DeliveryContent content) throws IOException {
		return serialize(content, true);
	}

	public static FreezableDeliveryContent discard(DeliveryContent content) throws IOException {
		return new FreezableDeliveryContent(content.withMessage(null), 0);
	}

	public static FreezableDeliveryContent copy(DeliveryContent content) throws IOException {
		return serialize(content, false);
	}

	private static FreezableDeliveryContent serialize(DeliveryContent content, boolean closeMessage)
			throws IOException {
		NamedChunk asChunk = ChunkFiler.newNamedChunk(MaxMessageSize.get());
		try (var buf = asChunk.chunk().appendStream(); @SuppressWarnings("deprecation")
		var hashedOutput = new HashingOutputStream(Hashing.sha1(), buf)) {
			Mime4JHelper.serialize(content.message(), hashedOutput);
			String messageGuid = hashedOutput.hash().toString();
			if (closeMessage) {
				closeMessage(content.message());
			}
			SerializedMessage serializedMessage = new SerializedMessage(asChunk, messageGuid);
			content.mailboxRecord().messageBody = messageGuid;
			return new FreezableDeliveryContent(content, serializedMessage);
		}
	}

	private static void closeMessage(Message message) {
		try {
			message.close();
		} catch (Exception e) {
			// This should not happen
		}
	}
}
