package net.bluemind.core.auditlogs.client.es.datastreams;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;

import co.elastic.clients.ApiClient;
import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch._types.ElasticsearchException;
import co.elastic.clients.elasticsearch._types.HealthStatus;
import co.elastic.clients.elasticsearch.ilm.get_lifecycle.Lifecycle;
import co.elastic.clients.elasticsearch.indices.get_index_template.IndexTemplateItem;
import co.elastic.clients.elasticsearch.indices.resolve_index.ResolveIndexDataStreamsItem;
import co.elastic.clients.transport.ElasticsearchTransport;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.auditlogs.IAuditLogMgmt;
import net.bluemind.core.auditlogs.client.es.AudiLogEsClientActivator;
import net.bluemind.core.auditlogs.client.es.datastreams.DataStreamActivator.ILMPolicyDefinition;
import net.bluemind.core.auditlogs.client.es.datastreams.DataStreamActivator.IndexTemplateDefinition;
import net.bluemind.core.auditlogs.client.loader.config.AuditLogConfig;
import net.bluemind.core.auditlogs.exception.AuditLogCreationException;
import net.bluemind.core.auditlogs.exception.AuditLogILMPolicyException;
import net.bluemind.core.auditlogs.exception.AuditLogRemovalException;

public class DataStreamService implements IAuditLogMgmt {
	private static Logger logger = LoggerFactory.getLogger(DataStreamService.class);

	/*
	 * Creates datastream and corresponding index template using audit log pattern
	 * name defined in configuration file auditlog-store.conf
	 */
	@Override
	public void setupAuditLogBackingStore(String domainUid) throws AuditLogCreationException {
		String dataStreamFullName = AuditLogConfig.resolveDataStreamName(domainUid);
		try {
			createDataStream(dataStreamFullName);
		} catch (ElasticsearchException | IOException e) {
			throw new AuditLogCreationException(e);
		}
	}

	/*
	 * Removes all datastreams and index templates
	 */
	@VisibleForTesting
	@Override
	public void removeAuditLogBackingStores() {
		ElasticsearchClient esClient = AudiLogEsClientActivator.get();
		Optional<IndexTemplateDefinition> optSchema = Optional
				.ofNullable(DataStreamActivator.getindexTemplateDefinition());
		// domainUId is not present, removes datastream root (e.g. audit_log*)
		String dataStreamName = "*";
		if (optSchema.isPresent()) {
			try {
				removeDataStream(esClient, dataStreamName);
			} catch (AuditLogRemovalException e) {
				logger.error("Error on audit log store removal: {}", e.getMessage());
			}
			try {
				removeIndexTemplate(esClient, optSchema.get().indexTemplateName());
			} catch (AuditLogRemovalException e) {
				logger.error("Error on audit log store removal: {}", e.getMessage());
			}
		}
	}

	@Override
	public void removeAuditLogBackingStore(String domainUid) {
		String dataStreamFullName = AuditLogConfig.resolveDataStreamName(domainUid);
		try {
			removeDataStream(dataStreamFullName);
		} catch (AuditLogRemovalException e) {
			logger.error("Failed to delete audit log store '{}': {}", dataStreamFullName, e.getMessage());
		}
	}

	@Override
	public boolean hasAuditLogBackingStore(String domainUid) {
		String dataStreamName = AuditLogConfig.resolveDataStreamName(domainUid);
		try {
			return hasDataStream(dataStreamName);
		} catch (ElasticsearchException | IOException e) {
			logger.error(e.getMessage());
			return false;
		}
	}

	@Override
	public void updateILMPolicyRetentionDuration(int duration) throws AuditLogILMPolicyException {
		try {
			ILMPolicyDefinition ilmPolicyDefinition = DataStreamActivator.getIlmPolicyDefinition();
			if (ilmPolicyDefinition != null) {
				String base64String = new String(ilmPolicyDefinition.schema());
				JsonObject jsonObject = new JsonObject(base64String);
				if (jsonObject.containsKey("policy") && jsonObject.getJsonObject("policy").containsKey("phases")
						&& jsonObject.getJsonObject("policy").getJsonObject("phases").containsKey("delete")) {
					jsonObject.getJsonObject("policy").getJsonObject("phases").getJsonObject("delete").put("min_age",
							duration + "d");
					ElasticsearchClient esClient = AudiLogEsClientActivator.get();
					esClient.ilm().putLifecycle(r -> r.name(ilmPolicyDefinition.name())
							.withJson(new ByteArrayInputStream(jsonObject.toString().getBytes())));
				}
			}
		} catch (ElasticsearchException | IOException e) {
			throw new AuditLogILMPolicyException(e.getMessage());
		}
	}

	@Override
	public String getRetentionDuration() throws AuditLogILMPolicyException {
		ILMPolicyDefinition ilmPolicyDefinition = DataStreamActivator.getIlmPolicyDefinition();
		if (ilmPolicyDefinition == null) {
			throw new AuditLogILMPolicyException("Cannot get ILM policy for auditlog");
		}

		ElasticsearchClient esClient = AudiLogEsClientActivator.get();
		try {
			Lifecycle lcr = esClient.ilm().getLifecycle().get(ilmPolicyDefinition.name());
			return lcr.policy().phases().delete().minAge().time();
		} catch (ElasticsearchException | IOException e) {
			throw new AuditLogILMPolicyException(
					"Cannot get duration for ilm '" + ilmPolicyDefinition.name() + "': " + e.getMessage());
		}

	}

	@Override
	public void removeDataStream(String dataStreamFullName) throws AuditLogRemovalException {
		ElasticsearchClient esClient = AudiLogEsClientActivator.get();
		Optional<IndexTemplateDefinition> optSchema = Optional
				.ofNullable(DataStreamActivator.getindexTemplateDefinition());
		if (optSchema.isPresent()) {
			removeDataStream(esClient, dataStreamFullName);
			removeIndexPatternFromIndexTemplate(esClient, optSchema.get(), dataStreamFullName + "*");
			removeIndexTemplate(esClient, optSchema.get().indexTemplateName());
		}
	}

	/*
	 * Creates datastream and index template using datastream full name defined as
	 * method argument
	 */
	public void createDataStream(String dataStreamFullName) throws IOException {
		ElasticsearchClient esClient = AudiLogEsClientActivator.get();
		List<String> currentDataStreams = dataStreamNames(esClient);
		Optional<IndexTemplateDefinition> optSchema = Optional
				.ofNullable(DataStreamActivator.getindexTemplateDefinition());
		if (!currentDataStreams.contains(dataStreamFullName) && optSchema.isPresent()) {
			updateILMPolicy(esClient, DataStreamActivator.getIlmPolicyDefinition());
			initOrUpdateIndexTemplate(esClient, optSchema.get(), dataStreamFullName);
			initDataStream(esClient, dataStreamFullName);
		}
	}

	private static void removeDataStream(ElasticsearchClient esClient, String dataStreamName)
			throws AuditLogRemovalException {
		Optional<IndexTemplateDefinition> optSchema = Optional
				.ofNullable(DataStreamActivator.getindexTemplateDefinition());
		if (optSchema.isPresent()) {
			try {
				boolean isDeleted = esClient.indices().deleteDataStream(d -> d.name(Arrays.asList(dataStreamName)))
						.acknowledged();
				logger.info("datastream '{}' deleted: {}.", dataStreamName, isDeleted);
			} catch (ElasticsearchException e) {
				if (e.error() != null && "index_not_found_exception".equals(e.error().type())) {
					logger.warn("dataStream '{}' not found, can't be delete", dataStreamName);
					return;
				}
				throw new AuditLogRemovalException(e);
			} catch (IOException e) {
				throw new AuditLogRemovalException(e);
			}
		}
	}

	public static List<String> dataStreamNames(ElasticsearchClient esClient) {
		try {
			return esClient.indices().resolveIndex(r -> r.name("*")).dataStreams() //
					.stream().map(ResolveIndexDataStreamsItem::name).toList();
		} catch (ElasticsearchException | IOException e) {
			logger.error("Failed to list datastreams", e);
			return Collections.emptyList();
		}
	}

	private static void removeIndexPatternFromIndexTemplate(ElasticsearchClient esClient,
			IndexTemplateDefinition indexTemplateDefinition, String indexPatternToRemove) {
		final String indexPatternField = "index_patterns";
		JsonObject jsonSchema = new JsonObject(new String(indexTemplateDefinition.schema()));
		JsonArray indexPatternsSchemaArray = new JsonArray();

		try {
			indexTemplateDefinitionOf(esClient, indexTemplateDefinition.indexTemplateName())
					.ifPresent(indexTemplate -> {
						List<String> indexPatterns = new ArrayList<>(indexTemplate.indexTemplate().indexPatterns());
						indexPatterns.remove(indexPatternToRemove);
						indexPatterns.forEach(a -> {
							if (!indexPatternsSchemaArray.contains(a)) {
								indexPatternsSchemaArray.add(a);
							}
						});
					});

			jsonSchema.put(indexPatternField, indexPatternsSchemaArray);
			byte[] enhancedSchema = jsonSchema.toString().getBytes();
			esClient.indices().putIndexTemplate(it -> it.name(indexTemplateDefinition.indexTemplateName())
					.withJson(new ByteArrayInputStream(enhancedSchema)));
			logger.info("Remove '{}'from index_patterns field for '{}' template", indexPatternToRemove,
					indexTemplateDefinition.indexTemplateName());
		} catch (IOException | ElasticsearchException e) {
			logger.error("error with removeIndexPatternFromIndexTemplate: {}", e.getMessage());
		}
	}

	private static void removeIndexTemplate(ElasticsearchClient esClient, String indexTemplateName)
			throws AuditLogRemovalException {
		try {
			esClient.indices().deleteIndexTemplate(it -> it.name(Arrays.asList(indexTemplateName)));
			logger.info("index template '{}' deleted.", indexTemplateName);
		} catch (ElasticsearchException e) {
			if (e.error() != null && "index_template_missing_exception".equals(e.error().type())) {
				logger.warn("index template '{}' not found, can't be delete", indexTemplateName);
				return;
			}
			if (e.error() != null && "illegal_argument_exception".equals(e.error().type())) {
				logger.warn("index template '{}' already in use, can't be delete: {}", indexTemplateName,
						e.getMessage());
				return;
			}
			throw new AuditLogRemovalException(e);
		} catch (IOException e) {
			throw new AuditLogRemovalException(e);
		}
	}

	private static void initDataStream(ElasticsearchClient esClient, String dataStreamName)
			throws ElasticsearchException, IOException {
		esClient.indices().createDataStream(d -> d.name(dataStreamName));
		logger.info("datastream '{}' created, waiting for green...", dataStreamName);
		esClient.cluster().health(h -> h.index(dataStreamName).waitForStatus(HealthStatus.Green));
	}

	private static void initOrUpdateIndexTemplate(ElasticsearchClient esClient,
			IndexTemplateDefinition indexTemplateDefinition, String indexPattern)
			throws ElasticsearchException, IOException {
		final String indexPatternField = "index_patterns";
		JsonObject jsonSchema = new JsonObject(new String(indexTemplateDefinition.schema()));

		// We need to change the index_patterns for the index_template definition, to
		// allow datastream association
		JsonArray indexPatternsSchemaArray = (!jsonSchema.containsKey(indexPatternField)) ? new JsonArray()
				: jsonSchema.getJsonArray(indexPatternField);
		logger.info("Update index_patterns field with '{}' for audit log index template", indexPattern);
		indexPatternsSchemaArray.add(indexPattern + "*");

		indexTemplateDefinitionOf(esClient, indexTemplateDefinition.indexTemplateName()).ifPresent(indexTemplate -> {
			// index template is present -> add indexPattern to index_patterns and updates
			List<String> indexPatterns = indexTemplate.indexTemplate().indexPatterns();
			indexPatterns.forEach(a -> {
				if (!indexPatternsSchemaArray.contains(a)) {
					indexPatternsSchemaArray.add(a);
				}
			});
		});

		jsonSchema.put(indexPatternField, indexPatternsSchemaArray);
		byte[] enhancedSchema = jsonSchema.toString().getBytes();
		esClient.indices().putIndexTemplate(it -> it.name(indexTemplateDefinition.indexTemplateName())
				.withJson(new ByteArrayInputStream(enhancedSchema)));
	}

	private static void updateILMPolicy(ElasticsearchClient esClient, ILMPolicyDefinition ilmPolicyDefinition)
			throws ElasticsearchException, IOException {
		if (ilmPolicyDefinition != null) {
			esClient.ilm().putLifecycle(r -> r.name(ilmPolicyDefinition.name())
					.withJson(new ByteArrayInputStream(ilmPolicyDefinition.schema())));
		}
	}

	public boolean hasDataStream(String dataStreamFullName) throws IOException {
		ElasticsearchClient esClient = AudiLogEsClientActivator.get();
		return !esClient.indices().resolveIndex(i -> i.name(dataStreamFullName)).dataStreams().isEmpty();
	}

	private static Optional<IndexTemplateItem> indexTemplateDefinitionOf(ElasticsearchClient esClient,
			String indexTemplateName) throws ElasticsearchException, IOException {
		return esClient.indices().getIndexTemplate().indexTemplates().stream()
				.filter(i -> i.name().equals(indexTemplateName)).findFirst();
	}

	public static <T extends ApiClient<?, ?>> T buildClient(String tag, ElasticsearchTransport transport,
			Function<ElasticsearchTransport, T> builder) {
		if (transport == null) {
			logger.error("no elasticsearch instance found for tag {}", tag);
			return null;
		} else {
			return builder.apply(transport);
		}
	}

}
