package net.bluemind.central.reverse.proxy.model;

import static net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.HEADER_ACTION;
import static net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.HEADER_TS;
import static net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.INFO_ADDR;
import static net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.TIME_MANAGE_WARN;
import static net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.TIME_PROCES_WARN;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import org.apache.curator.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.curator.shaded.com.google.common.util.concurrent.RateLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Vertx;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageConsumer;
import net.bluemind.central.reverse.proxy.model.common.DirInfo;
import net.bluemind.central.reverse.proxy.model.common.DirInfo.DirEmail;
import net.bluemind.central.reverse.proxy.model.common.DomainInfo;
import net.bluemind.central.reverse.proxy.model.common.InstallationInfo;
import net.bluemind.central.reverse.proxy.model.common.ProxyInfoStoreEventBusAddress.ActionHeader;

public class ProxyInfoStore {
	private final Logger logger = LoggerFactory.getLogger(ProxyInfoStore.class);
	private final RateLimiter rlLogLag = RateLimiter.create(1);

	private final ProxyInfoStorage storage;

	private MessageConsumer<Object> consumer;

	private ProxyInfoStore(ProxyInfoStorage storage) {
		this.storage = storage;
	}

	public static ProxyInfoStore create() {
		return ProxyInfoStore.create(ProxyInfoStorage.create());
	}

	public static ProxyInfoStore create(ProxyInfoStorage storage) {
		return new ProxyInfoStore(storage);
	}

	public ProxyInfoStore setupService(Vertx vertx) {
		consumer = vertx.eventBus().<Object>consumer(INFO_ADDR).handler(event -> {
			logEventProcessDuration(event);
			long start = System.nanoTime();

			ActionHeader action = ActionHeader.fromString(event.headers().get(HEADER_ACTION));
			if (action == ActionHeader.ALL_IPS) {
				allIps(event);
			} else {
				if (event.body() == null) {
					switch (action) {
					case IP -> ip(event, null);
					case ANY_IP -> anyIp(event);
					default -> event.fail(404, "Unknown action '" + action + "'");
					}
				} else {
					switch (event.body()) {
					case DirInfo dir -> addDir(event, dir);
					case DomainInfo domain -> addDomain(event, domain);
					case InstallationInfo installation -> addInstallation(event, installation);
					case String msg -> ip(event, msg);
					default -> event.fail(404, "Unknown action '" + action + "'");
					}
				}
			}

			long processedTime = System.nanoTime() - start;
			if (logger.isDebugEnabled()) {
				logger.debug("ProxyInfoStore: vertx event management took {}ms long", processedTime / 1000);
			} else if (processedTime > TIME_MANAGE_WARN && rlLogLag.tryAcquire()) {
				logger.warn("ProxyInfoStore: vertx event management took more than {}ms long: {}ms",
						TIME_MANAGE_WARN / 1000, processedTime / 1000);
			}
		});

		return this;
	}

	private void logEventProcessDuration(Message<Object> event) {
		try {
			long ts = Long.parseLong(event.headers().get(HEADER_TS));

			long processTime = System.nanoTime() - ts;
			if (processTime > TIME_PROCES_WARN && rlLogLag.tryAcquire()) {
				logger.warn("ProxyInfoStore: vertx event process took more than {}ms long: {}ms",
						TIME_PROCES_WARN / 1000, processTime / 1000);
			}
		} catch (NumberFormatException nfe) {
			// Ignore bad event timestamp
		}
	}

	private List<String> expand(DirEmail email, String domainUid) {
		if (email.allAliases) {
			Set<String> domainAliases = storage.domainAliases(domainUid);
			domainAliases.add(domainUid);
			String leftPart = email.address.split("@")[0];
			return domainAliases.stream().map(alias -> leftPart + "@" + alias).toList();
		} else {
			return Collections.singletonList(email.address);
		}
	}

	private void addDir(Message<Object> event, DirInfo dir) {
		if (dir.kind == null || !dir.kind.equalsIgnoreCase("user") || dir.emails.isEmpty()) {
			event.reply(null);
		}
		dir.emails.stream() //
				.flatMap(email -> expand(email, dir.domainUid).stream()) //
				.forEach(email -> storage.addLogin(email, dir.dataLocation));
		event.reply(null);
	}

	private void addDomain(Message<Object> event, DomainInfo domain) {
		storage.addDomain(domain.uid, domain.aliases);
		event.reply(null);
	}

	private void addInstallation(Message<Object> event, InstallationInfo installation) {
		if (!installation.hasNginx) {
			event.reply(null);
		}

		String oldIp = storage.addDataLocation(installation.dataLocationUid, installation.ip);
		event.reply(oldIp);
	}

	private void ip(Message<Object> event, String login) {
		if (Objects.isNull(login)) {
			event.fail(500, "unable to decode parameters '" + event.body() + "'");
		} else {
			String ip = storage.ip(login);
			if (!Objects.isNull(ip)) {
				event.reply(ip);
			} else {
				event.fail(404, "No IP found for '" + login + "'");
			}
		}
	}

	private void anyIp(Message<Object> event) {
		String anyIp = storage.anyIp();
		if (!Objects.isNull(anyIp)) {
			event.reply(anyIp);
		} else {
			event.fail(404, "No IPs found");
		}
	}

	private void allIps(Message<Object> event) {
		event.reply(storage.allIps());
	}

	@VisibleForTesting
	public void tearDown() throws InterruptedException, ExecutionException {
		if (consumer != null) {
			CompletableFuture<Void> c = new CompletableFuture<>();
			consumer.unregister().onSuccess(c::complete).onFailure(c::completeExceptionally);
			c.get();
		}
	}

}
