package net.bluemind.central.reverse.proxy.model;

import static net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.ADDRESS;
import static net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.HEADER_ACTION;
import static net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.HEADER_TS;
import static net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.TIME_MANAGE_WARN;
import static net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.TIME_PROCES_WARN;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import org.apache.curator.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.curator.shaded.com.google.common.base.Strings;
import org.apache.curator.shaded.com.google.common.util.concurrent.RateLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Vertx;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageConsumer;
import net.bluemind.central.reverse.proxy.model.common.DirInfo;
import net.bluemind.central.reverse.proxy.model.common.DomainInfo;
import net.bluemind.central.reverse.proxy.model.common.DomainSettings;
import net.bluemind.central.reverse.proxy.model.common.InstallationInfo;
import net.bluemind.central.reverse.proxy.model.common.MemberInfo;
import net.bluemind.central.reverse.proxy.model.common.PostfixMapsStoreEventBusAddress.PostfixActionHeader;

public class PostfixMapsStore {
	private final Logger logger = LoggerFactory.getLogger(PostfixMapsStore.class);
	private final RateLimiter rlLogLag = RateLimiter.create(1);

	private final PostfixMapsStorage storage;

	private MessageConsumer<Object> consumer;

	private PostfixMapsStore(PostfixMapsStorage storage) {
		this.storage = storage;
	}

	public static PostfixMapsStore create() {
		return PostfixMapsStore.create(PostfixMapsStorage.create());
	}

	public static PostfixMapsStore create(PostfixMapsStorage storage) {
		return new PostfixMapsStore(storage);
	}

	public PostfixMapsStore setupService(Vertx vertx) {
		consumer = vertx.eventBus().<Object>consumer(ADDRESS).handler(event -> {
			logEventProcessDuration(event);
			long start = System.nanoTime();
			PostfixActionHeader action = PostfixActionHeader.fromString(event.headers().get(HEADER_ACTION));

			switch (event.body()) {
			case InstallationInfo installationInfo -> addInstallation(event, installationInfo);
			case DirInfo dirInfo -> addDir(event, dirInfo);
			case DomainInfo domainInfo -> {
				switch (action) {
				case DEL_DOMAIN -> delDomain(event, domainInfo);
				case ADD_DOMAIN -> addDomain(event, domainInfo);
				default -> event.fail(404, "Unknown action '" + action + "'");
				}
			}
			case DomainSettings domainSettings -> updateDomainSettings(event, domainSettings);
			case MemberInfo memberInfo -> manageMember(event, memberInfo);
			case String msg -> {
				switch (action) {
				case DEL_DIR -> delDir(event, msg);
				case ALIAS_TO_MAILBOX -> aliasToMailboxes(event, msg);
				case MAILBOX_EXISTS -> mailboxExists(event, msg);
				case MAILBOX_DOMAIN_MANAGED -> mailboxDomainManaged(event, msg);
				case MAILBOX_STORE -> mailboxRelay(event, msg);
				case SRS_RECIPIENT -> srsRecipient(event, msg);
				default -> event.fail(404, "Unknown action '" + action + "'");
				}
			}
			default -> event.fail(404, "Unknown action '" + action + "'");
			}

			long processedTime = System.nanoTime() - start;
			if (logger.isDebugEnabled()) {
				logger.debug("PostfixMapsStore: vertx event consumption took {}ms long",
						Duration.ofNanos(processedTime).toMillis());
			} else if (processedTime > TIME_MANAGE_WARN && rlLogLag.tryAcquire()) {
				logger.warn("PostfixMapsStore: vertx event consumption took more than {}ms long: {}ms",
						TIME_MANAGE_WARN / 1000, Duration.ofNanos(processedTime).toMillis());
			}
		});

		return this;
	}

	private void logEventProcessDuration(Message<Object> event) {
		try {
			long ts = Long.parseLong(event.headers().get(HEADER_TS));

			long processTime = System.nanoTime() - ts;
			if (processTime > TIME_PROCES_WARN && rlLogLag.tryAcquire()) {
				logger.warn("PostfixMapsStore: vertx event process took more than {}ms long: {}ms",
						TIME_PROCES_WARN / 1000, Duration.ofNanos(processTime).toMillis());
			}
		} catch (NumberFormatException nfe) {
			// Ignore bad event timestamp
		}
	}

	private void addInstallation(Message<Object> event, InstallationInfo installationInfo) {
		if (!installationInfo.hasCore) {
			event.reply(null);
			return;
		}
		storage.updateInstallationUid(installationInfo.uid);
		storage.updateLmtpIP(installationInfo.ip);
		event.reply(null);
	}

	private void addDir(Message<Object> event, DirInfo dir) {
		if (dir.archived) {
			storage.removeUid(dir.entryUid);
			event.reply(null);
			return;
		}

		if (dir.kind.equalsIgnoreCase("group")) {
			if (dir.routing.equalsIgnoreCase("internal")) {
				storage.updateMailbox(dir.domainUid, dir.entryUid, getDirMailboxName(dir), dir.routing,
						dir.dataLocation);
				storage.addRecipient(dir.entryUid, "group-archive", dir.entryUid);
			} else {
				storage.removeRecipient(dir.entryUid, "group-archive", dir.entryUid);
				storage.removeMailbox(dir.entryUid);
			}
		} else if (!Strings.isNullOrEmpty(dir.mailboxName) && !Strings.isNullOrEmpty(dir.routing)
				&& !Strings.isNullOrEmpty(dir.dataLocation)) {
			storage.updateMailbox(dir.domainUid, dir.entryUid, getDirMailboxName(dir), dir.routing, dir.dataLocation);
		}
		storage.updateEmails(dir.entryUid, dir.emails);
		event.reply(null);
	}

	private void delDir(Message<Object> event, String entryUid) {
		storage.removeUid(entryUid);
		event.reply(null);
	}

	private String getDirMailboxName(DirInfo dir) {
		return (dir.kind.equalsIgnoreCase("user") ? "" : "+") + dir.mailboxName + "@" + dir.domainUid;
	}

	private void addDomain(Message<Object> event, DomainInfo domain) {
		storage.updateDomain(domain.uid, domain.aliases);
		event.reply(null);
	}

	private void updateDomainSettings(Message<Object> event, DomainSettings domainSettings) {
		storage.updateDomainSettings(domainSettings.domainUid, domainSettings.mailRoutingRelay,
				domainSettings.mailForwardUnknown);
		event.reply(null);
	}

	private void delDomain(Message<Object> event, DomainInfo domain) {
		storage.removeDomain(domain.uid);
		event.reply(null);
	}

	private void manageMember(Message<Object> event, MemberInfo member) {
		if (member.added) {
			storage.addRecipient(member.groupUid, member.memberType, member.memberUid);
		} else {
			storage.removeRecipient(member.groupUid, member.memberType, member.memberUid);
		}
		event.reply(null);
	}

	private void aliasToMailboxes(Message<Object> event, String email) {
		event.reply(storage.aliasToMailboxes(email));
	}

	private void mailboxExists(Message<Object> event, String mailbox) {
		event.reply(Boolean.valueOf(storage.mailboxManaged(mailbox)));
	}

	private void mailboxDomainManaged(Message<Object> event, String mailboxDomain) {
		event.reply(storage.domainManaged(mailboxDomain));
	}

	private void mailboxRelay(Message<Object> event, String mailbox) {
		event.reply(storage.mailboxRelay(mailbox));
	}

	private void srsRecipient(Message<Object> event, String recipient) {
		event.reply(storage.srsRecipient(recipient));
	}

	@VisibleForTesting
	public void tearDown() throws InterruptedException, ExecutionException {
		if (consumer != null) {
			CompletableFuture<Void> c = new CompletableFuture<>();
			consumer.unregister().onSuccess(c::complete).onFailure(c::completeExceptionally);
			c.get();
		}
	}
}
