/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.sds.store;

import java.lang.ref.Reference;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.typesafe.config.Config;

import io.vertx.core.Vertx;
import io.vertx.core.json.JsonObject;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.ExistResponse;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.MgetRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.dto.TierMoveRequest;
import net.bluemind.sds.dto.TierMoveResponse;
import net.bluemind.system.api.ArchiveKind;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;

public interface ISdsBackingStoreFactory {

	public ArchiveKind kind();

	public ISdsBackingStore create(Vertx vertx, JsonObject configuration, String dataLocation);

	public default ISdsBackingStore create(Vertx vertx, SystemConf sysconf, String dataLocation) {
		JsonObject jsonconf = new JsonObject()//
				.put("storeType", sysconf.stringValue(SysConfKeys.archive_kind.name()))//
				.put("endpoint", sysconf.stringValue(SysConfKeys.sds_s3_endpoint.name()))//
				.put("accessKey", sysconf.stringValue(SysConfKeys.sds_s3_access_key.name()))//
				.put("secretKey", sysconf.stringValue(SysConfKeys.sds_s3_secret_key.name()))//
				.put("region", sysconf.stringValue(SysConfKeys.sds_s3_region.name()))//
				.put("bucket", sysconf.stringValue(SysConfKeys.sds_s3_bucket.name()))//
				.put("insecure", sysconf.booleanValue(SysConfKeys.sds_s3_insecure.name(), false))
				.put("useSplitPath", sysconf.booleanValue(SysConfKeys.sds_s3_split_path.name(), false));
		return this.create(vertx, jsonconf, dataLocation);
	}

	default ISdsSyncStore createSync(Vertx vertx, SystemConf sysconf, String dataLocation) {
		return syncStore(create(vertx, sysconf, dataLocation));
	}

	default ISdsSyncStore syncStore(ISdsBackingStore asyncStore) {
		return new ISdsSyncStore() {
			Config coreConfig = CoreConfig.get();
			Duration readTimeout = coreConfig.getDuration(CoreConfig.SDS.READ_TIMEOUT);
			Duration writeTimeout = coreConfig.getDuration(CoreConfig.SDS.WRITE_TIMEOUT);

			@Override
			public ExistResponse exists(ExistRequest req) {
				try {
					return asyncStore.exists(req).get(readTimeout.toSeconds(), TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("exists got interrupted", ErrorCode.UNKNOWN);
				} catch (ExecutionException e) {
					throw new ServerFault("exists failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("exists timed out", ErrorCode.TIMEOUT);
				}
			}

			@Override
			public SdsResponse upload(PutRequest req) {
				try {
					return asyncStore.upload(req).get(writeTimeout.toSeconds(), TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("upload got interrupted", ErrorCode.UNKNOWN, e);
				} catch (ExecutionException e) {
					throw new ServerFault("upload failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("upload timed out", ErrorCode.TIMEOUT);
				} finally {
					Reference.reachabilityFence(req.filename);
				}
			}

			@Override
			public SdsResponse download(GetRequest req) {
				try {
					return asyncStore.download(req).get(readTimeout.toSeconds(), TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("download got interrupted", ErrorCode.UNKNOWN);
				} catch (ExecutionException e) {
					throw new ServerFault("download failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("download timed out", ErrorCode.TIMEOUT);
				} finally {
					Reference.reachabilityFence(req.filename);
				}
			}

			@Override
			public SdsResponse downloadRaw(GetRequest req) {
				try {
					return asyncStore.downloadRaw(req).get(readTimeout.toSeconds(), TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("download got interrupted", ErrorCode.UNKNOWN);
				} catch (ExecutionException e) {
					throw new ServerFault("download failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("download timed out", ErrorCode.TIMEOUT);
				} finally {
					Reference.reachabilityFence(req.filename);
				}
			}

			@Override
			public SdsResponse downloads(MgetRequest req) {
				try {
					// We do a multiplication here, because we expect many requests
					return asyncStore.downloads(req).get(readTimeout.toSeconds() * 2, TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("downloads got interrupted", ErrorCode.UNKNOWN);
				} catch (ExecutionException e) {
					throw new ServerFault("downloads failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("downloads timed out", ErrorCode.TIMEOUT);
				} finally {
					for (var t : req.transfers) {
						Reference.reachabilityFence(t.filename);
					}
				}
			}

			@Override
			public SdsResponse delete(DeleteRequest req) {
				try {
					return asyncStore.delete(req).get(writeTimeout.toSeconds(), TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("delete got interrupted", ErrorCode.UNKNOWN);
				} catch (TimeoutException e) {
					throw new ServerFault("delete timed out", ErrorCode.TIMEOUT, e);
				} catch (ExecutionException e) {
					throw new ServerFault("delete failed", ErrorCode.TIMEOUT, e.getCause());
				}
			}

			@Override
			public TierMoveResponse tierMove(TierMoveRequest tierMoveRequest) {
				try {
					return asyncStore.tierMove(tierMoveRequest).get(60, TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					throw new ServerFault("tier move got interrupted", ErrorCode.UNKNOWN);
				} catch (ExecutionException e) {
					throw new ServerFault("tier move failed", ErrorCode.UNKNOWN, e);
				} catch (TimeoutException e) {
					throw new ServerFault("tier move timed out", ErrorCode.TIMEOUT);
				}
			}

			@Override
			public void close() {
				asyncStore.close();
			}

			@Override
			public String toString() {
				return "ISdsSyncStore{store=" + asyncStore + "}";
			}
		};
	}

}
