import app from "@bluemind/global.app";
import store from "@bluemind/store";

import FileHostingModalContent from "../components/FileHostingModalContent.vue";
import getContentWithLinks from "../helpers/getContentWithLinks";
import {
    renderMustDetachConfirmBox,
    renderShouldDetachConfirmBox,
    renderTooLargeFilesModal
} from "../helpers/renderers";

import { ADD_FH_ATTACHMENT, GET_CONFIGURATION } from "../store/types/actions";

import { StopExecutionError } from "./errors";

let autoDetachmentLimit, maxFilesize;

export default async function ({ files, message }, { forceFilehosting }) {
    const maxMessageSize = store.state.mail.messageCompose.maxMessageSize;
    files = [...files];
    if (!autoDetachmentLimit || !maxFilesize) {
        ({ autoDetachmentLimit, maxFilesize } = await store.dispatch(`mail/${GET_CONFIGURATION}`));
    }
    const newAttachmentsSize = getFilesSize(files);
    if (maxFilesize && files.some(file => file.size > maxFilesize)) {
        renderTooLargeFilesModal(this, files, maxFilesize);
        throw new StopExecutionError();
    } else if (forceFilehosting) {
        return doDetach.call(this, files, message);
    } else if (message.size + newAttachmentsSize > maxMessageSize) {
        return mustDetachFiles.call(this, files, message, maxMessageSize);
    } else if (autoDetachmentLimit && newAttachmentsSize > autoDetachmentLimit) {
        return shouldDetachFiles.call(this, files, message, maxMessageSize);
    }
}

async function mustDetachFiles(files, message, maxMessageSize) {
    const { content, props } = renderMustDetachConfirmBox(files, maxMessageSize, message);
    const res = await app.config.globalProperties.$modalConfirm.confirm({
        content,
        title: props.title,
        okLabel: props.okTitle,
        cancelLabel: props.cancelLabel,
        size: "md"
    });
    if (res) {
        return doDetach.call(this, files, message);
    }
    throw new StopExecutionError();
}

async function shouldDetachFiles(files, message) {
    const { content, props } = renderShouldDetachConfirmBox(files);
    const res = await app.config.globalProperties.$modalConfirm.confirm({
        content,
        title: props.title,
        okLabel: props.okTitle,
        cancelLabel: props.cancelTitle,
        size: "md"
    });
    if (res) {
        return doDetach.call(this, files, message);
    }
}

async function doDetach(files, message) {
    try {
        app.config.globalProperties.$modalDialog.open(FileHostingModalContent, {
            data: { sizeLimit: store.state.mail.messageCompose.maxMessageSize, message }
        });
        await Promise.all(files.map(file => store.dispatch(`mail/${ADD_FH_ATTACHMENT}`, { file, message })));
        const newContent = getContentWithLinks(message);
        store.dispatch("mail/SET_DRAFT_CONTENT", { html: newContent, draft: message });
    } catch (e) {
        // eslint-disable-next-line no-console
        console.warn(e);
    }
    return { files: [], message };
}

function getFilesSize(files) {
    return files.reduce((totalSize, next) => totalSize + next.size, 0);
}
