/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.metrics.core.healing;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NCUtils;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.TagDescriptor;

public class NginxPeriodicHealing extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(NginxPeriodicHealing.class);

	@Override
	public void start() throws Exception {
		VertxPlatform.executeBlockingPeriodic(TimeUnit.HOURS.toMillis(2), this::reloadNginx);
	}

	private void reloadNginx(long tid) {
		Topology.getIfAvailable().ifPresent(
				t -> t.all(TagDescriptor.bm_nginx.getTag(), TagDescriptor.bm_nginx_edge.getTag()).forEach(nginx -> {
					INodeClient node = NodeActivator.get(nginx.value.address());
					NCUtils.execNoOut(node, List.of("systemctl", "reload", "bm-nginx"), 5, TimeUnit.SECONDS);
					logger.info("Nginx {} reloaded to avoid mem-leaks", nginx.value);
				}));
	}

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {

		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new NginxPeriodicHealing();
		}

	}

}
