/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2025
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.cluster.common;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatch.State;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NCUtils;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.Server;

public class CuratorLatchManager {

	private static final Logger logger = LoggerFactory.getLogger(CuratorLatchManager.class);
	private static ConcurrentHashMap<String, LeaderLatch> latches = new ConcurrentHashMap<>();
	private static final String CLUSTER_ID = getIdFromFile("/etc/bm/mcast.id");
	private static final String PARTICIPANT_ID = getParticipantId("/etc/bm/server.uid");

	private static final ILeadershipHandle NOT_IN_CLUSTER = () -> true;

	private CuratorLatchManager() {

	}

	public static ILeadershipHandle startLatch(String latchName, ILeadershipListener listener) {
		if (ClusterCuratorClient.getFramework().isPresent()) {
			logger.info("Starting latch as \"{}\" on \"{}\", single core = {}", PARTICIPANT_ID, getLatchPath(latchName),
					Topology.get().singleCore());
			LeaderLatch latch = getLeaderLatch(latchName);
			try {
				latch.start();
				latch.addListener(simpleLatchListener(listener));
				return () -> latch.hasLeadership() || Topology.get().singleCore();
			} catch (Exception e) {
				throw new ServerFault(e);
			}
		} else if (Topology.get().singleCore()) {
			return NOT_IN_CLUSTER;
		} else {
			throw new ServerFault("Cannot start curator latch " + latchName);
		}
	}

	private static LeaderLatchListener simpleLatchListener(ILeadershipListener listener) {
		return new LeaderLatchListener() {

			@Override
			public void isLeader() {
				listener.obtainedLeadership();
			}

			@Override
			public void notLeader() {
				listener.lostLeadership(() -> {
					// Seppuku handle
					ItemValue<Server> myself = Topology.get().datalocation(net.bluemind.config.DataLocation.current());
					INodeClient nc = NodeActivator.get(myself.value.address());
					NCUtils.execNoOut(nc, List.of("systemctl", "restart", "bm-core"), 1, TimeUnit.MINUTES);
				});
			}
		};
	}

	static void closeLatches() {
		logger.info("Closing all latches");
		ClusterCuratorClient.getFramework().ifPresent(f -> {
			latches.forEach((name, latch) -> {
				try {
					if (latch != null && latch.getState() != State.CLOSED) {
						latch.close();
					}
				} catch (IllegalStateException e) {
					if (!e.getMessage().startsWith("Expected state [STARTED] was [STOPPED]")) {
						logger.error("Exception while closing latch \"" + name + "\"", e);
					}
				} catch (IOException e) {
					logger.error("Exception while closing latch \"" + name + "\"", e);
				}
			});
		});
		latches.clear();
	}

	private static LeaderLatch getLeaderLatch(String latchName) {
		return latches.computeIfAbsent(latchName, k -> {
			return new LeaderLatch(ClusterCuratorClient.getFramework().get(), getLatchPath(latchName), PARTICIPANT_ID);
		});
	}

	private static String getLatchPath(String latchName) {
		return "/" + CLUSTER_ID + "/latches/" + latchName;
	}

	private static String getParticipantId(String serverUidFile) {
		return getIdFromFile(serverUidFile) + "-" + System.getProperty("net.bluemind.property.product", "unknown");
	}

	private static String getIdFromFile(String path) {
		File file = new File(path);
		if (file.exists()) {
			try {
				return Files.readString(file.toPath(), Charset.defaultCharset());
			} catch (IOException e) {
				logger.warn("Cannot read " + file.getName(), e);
			}
		}
		String randomId = UUID.randomUUID().toString();
		logger.warn("Cannot read {}, using {} instead.", file.getName(), randomId);
		return randomId;
	}

}
