/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2025
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.cluster.common;

import java.io.File;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.retry.BoundedExponentialBackoffRetry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import net.bluemind.network.topology.Topology;

public class ClusterCuratorClient {

	private static final Logger logger = LoggerFactory.getLogger(ClusterCuratorClient.class);
	private static String zkServer = zkServerFromConfig();
	private static CuratorFramework curator;

	private static String zkServerFromConfig() {
		Config conf = ConfigFactory.systemProperties();
		try {
			File local = new File("/etc/bm/cluster-zookeeper.conf"); // NOSONAR
			if (local.exists()) {
				Config parsed = ConfigFactory.parseFile(local);
				conf = conf.withFallback(parsed);
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		return conf.hasPath("cluster.zk.servers") ? conf.getString("cluster.zk.servers") : null;
	}

	public static synchronized Optional<CuratorFramework> getFramework() {
		if (zkServer == null) {
			return Optional.empty();
		}

		if (curator != null) {
			return Optional.of(curator);
		}

		logger.info("Connecting to zookeeper");
		BoundedExponentialBackoffRetry rtp = new BoundedExponentialBackoffRetry(100, 10000, 15);
		curator = CuratorFrameworkFactory.newClient(zkServer, rtp);
		curator.start();
		try {
			if (Topology.getIfAvailable().map(t -> t.singleCore()).orElse(true)) {
				// If we are in mono-core context, zookeeper is optional even when configured
				if (!curator.blockUntilConnected(1, TimeUnit.SECONDS)) {
					// We could let curator continue to try connecting in the background but that
					// would probably pollute the logs for nothing
					disconnect();
				}
			} else {
				curator.blockUntilConnected();
			}
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			disconnect();
		}

		if (curator != null) {
			curator.getConnectionStateListenable().addListener(new ConnectionStateListener() {

				@Override
				public void stateChanged(CuratorFramework client, ConnectionState newState) {
					if (newState.isConnected()) {
						logger.info("Curator client state changed: {}", newState.name());
					} else {
						logger.warn("Curator client state changed: {}", newState.name());
						if (Topology.get().singleCore()) {
							// We could let curator continue to try connecting in the background but that
							// would probably pollute the logs for nothing
							disconnect();
						} else {
							// TODO Should we fire an event ? Restart ?
						}
					}
				}
			});
		}

		logger.info("Connection status: {}", curator == null ? "not connected" : curator.getState().toString());
		return Optional.ofNullable(curator);
	}

	protected static void disconnect() {
		if (curator != null) {
			logger.info("Terminating connection");
			curator.close();
			curator = null;
		}
	}

}
