/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.model.Container;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.user.repository.IUserSubscriptionStore;

public class CqlUserSubscription extends CqlAbstractStore implements IUserSubscriptionStore {

	private final CqlContainerStore contStore;

	public CqlUserSubscription(CqlSession s, CqlContainerStore conts) {
		super(s);
		this.contStore = conts;
	}

	@Override
	public void subscribe(String subject, Container container) throws SQLException {
		voidCql("""
				INSERT INTO t_subs_by_subject
				(subject, container_id, type, sync, mount)
				VALUES
				(?, ?, ?, ?, ?)
				""", subject, container.id, container.type, true, true);
	}

	@Override
	public boolean isSubscribed(String subject, Container container) throws SQLException {
		Long gotIt = unique("SELECT container_id FROM t_subs_by_subject WHERE subject=? AND container_id=?",
				r -> r.getLong(0), voidPop(), subject, container.id);
		return gotIt != null;
	}

	@Override
	public void unsubscribe(String subject, String containerUid) throws SQLException {
		Container loaded = contStore.get(containerUid);
		if (loaded != null) {
			voidCql("DELETE FROM t_subs_by_subject WHERE subject=? AND container_id=?", subject, loaded.id);
		}
	}

	@Override
	public void unsubscribeAll(String subject) throws SQLException {
		voidCql("DELETE FROM t_subs_by_subject WHERE subject=?");
	}

	@Override
	public List<String> listSubscriptions(String subject, String type) throws SQLException {
		List<Long> contIds = map("""
				SELECT container_id
				FROM t_subs_by_subject_and_type
				WHERE subject=? AND type=?
				""", r -> r.getLong(0), voidPop(), subject, type);
		return contStore.mgetUids(contIds);
	}

	@Override
	public List<String> subscribers(String containerUid) throws SQLException {
		Container cont = contStore.get(containerUid);
		return map("""
				SELECT subject
				FROM t_subs_by_container
				WHERE container_id=?
				""", r -> r.getString(0), voidPop(), cont.id);
	}

	@Override
	public void allowSynchronization(String subject, Container container, boolean sync) throws SQLException {
		voidCql("""
				UPDATE t_subs_by_subject
				SET sync = ?
				WHERE subject=? AND container_id=?
				""", sync, subject, container.id);

	}

	@Override
	public boolean isSyncAllowed(String subject, Container container) throws SQLException {
		Boolean res = unique("""
				SELECT sync
				FROM t_subs_by_subject
				WHERE subject=? AND container_id=?
				""", r -> r.getBoolean(0), voidPop(), subject, container.id);
		return res != null && res.booleanValue();
	}

	@Override
	public void updateAutomount(String subject, Container container, boolean automount) throws SQLException {
		voidCql("""
				UPDATE t_subs_by_subject
				SET mount = ?
				WHERE subject=? AND container_id=?
				""", automount, subject, container.id);
	}

	@Override
	public boolean isAutomounted(String subject, Container container) throws SQLException {
		Boolean res = unique("""
				SELECT mount
				FROM t_subs_by_subject
				WHERE subject=? AND container_id=?
				""", r -> r.getBoolean(0), voidPop(), subject, container.id);
		return res != null && res.booleanValue();
	}

}
