/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.repository.IContainerPersonalSettingsStore;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlPersonalSettingsStore extends CqlAbstractStore implements IContainerPersonalSettingsStore {

	private final String subject;
	private final long cid;

	public CqlPersonalSettingsStore(CqlSession s, String subject, Container c) {
		super(s);
		this.cid = c.id;
		this.subject = subject;
	}

	@Override
	public Map<String, String> get() throws SQLException {
		Map<String, String> stored = unique(
				"SELECT settings FROM t_container_owner_settings WHERE owner=? AND container_id=?",
				r -> r.getMap(0, String.class, String.class), voidPop(), subject, cid);
		return Optional.ofNullable(stored).orElseGet(Collections::emptyMap);
	}

	@Override
	public void set(Map<String, String> settings) throws SQLException {
		voidCql("UPDATE t_container_owner_settings SET settings=? WHERE owner=? AND container_id=?", settings, subject,
				cid);
	}

	@Override
	public void deleteAll() throws SQLException {
		voidCql("DELETE FROM t_container_owner_settings WHERE owner=? AND container_id!?", subject, cid);
	}

}
