/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.shaded.guava.common.collect.Lists;

import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IItemStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.cql.CqlFailedConstraintException;
import net.bluemind.cql.CqlPersistenceException;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.repository.sequences.ISequenceStore;
import net.bluemind.repository.sequences.Sequences;

public class CqlItemStore extends CqlAbstractStore implements IItemStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlItemStore.class);
	private final ISequenceStore seqs;
	private final Container container;
	private final SecurityContext ctx;

	public CqlItemStore(Container c, CqlSession s, ISequenceStore seq, SecurityContext ctx) {
		super(s);
		this.seqs = seq;
		this.container = c;
		this.ctx = ctx;
	}

	@Override
	public Item create(Item it) throws SQLException {
		String sn = Sequences.ownerIds(container.owner);
		if (it.id == 0) {
			it.id = seqs.nextVal(sn);
		} else {
			// ensure id was allocated
			long curValue = seqs.curVal(sn);
			if (it.id > curValue) {
				throw new CqlPersistenceException(
						"curValue of '" + sn + "' is " + curValue + ", " + it.id + " is not a usable id.");
			}
		}

		if (it.version == 0) {
			it.version = nextVersion();
		}
		it.updated = new Date();
		it.created = it.updated;
		BoundStatement bound = session//
				.prepare(
						"""
								INSERT INTO t_container_item
								(id, container_id, uid, version, external_id, display_name, createdby, updatedby, created, updated,
								flag_seen, flag_deleted, flag_important)
								VALUES
								(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
								IF NOT EXISTS
								""")
				.bind(it.id, container.id, it.uid, (int) it.version, it.externalId, it.displayName, it.createdBy,
						it.updatedBy, it.created.toInstant(), it.updated.toInstant(), it.flags.contains(ItemFlag.Seen),
						it.flags.contains(ItemFlag.Deleted), it.flags.contains(ItemFlag.Important));

		ResultSet rs = session.execute(bound);
		if (!rs.wasApplied()) {
			throw new CqlFailedConstraintException("Could not insert item " + it.uid);
		}

		if (it.unseen()) {
			updateUnseenCount(+1);
			if (it.visible()) {
				updateUnseenVisibleCount(+1);
			}
		}
		updateTotalCount(+1);
		if (logger.isDebugEnabled()) {
			logger.debug("Created {} in {}", it, container);
		}
		return it;
	}

	private long itemId(String uid) {
		PreparedStatement prep = session.prepare("""
				SELECT
				id
				FROM t_container_item_by_uid
				WHERE
				container_id = ? AND uid = ?
				""");
		ResultSet result = session.execute(prep.bind(container.id, uid));
		Row r = result.one();
		if (r == null) {
			return 0;
		}
		return r.getLong(0);
	}

	private long nextVersion() {
		return seqs.nextVal(Sequences.itemVersions(container.uid));
	}

	private void updateUnseenCount(long delta) {
		voidCql("UPDATE cnt_items_unseen SET unseen = unseen + ? WHERE container_id = ?", delta, container.id);
	}

	private void updateUnseenVisibleCount(long delta) {
		voidCql("UPDATE cnt_items_unseen_visible SET unseen = unseen + ? WHERE container_id = ?", delta, container.id);
	}

	private void updateTotalCount(long delta) {
		voidCql("UPDATE cnt_items_all SET all = all + ? WHERE container_id = ?", delta, container.id);
	}

	@Override
	public long count(ItemFlagFilter filter) throws SQLException {

		if (filter.matchUnseen()) {
			return unique("SELECT unseen FROM cnt_items_unseen WHERE container_id=?", r -> r.getLong(0), voidPop(),
					container.id);
		}

		if (filter.matchAll()) {
			return unique("SELECT all FROM cnt_items_all WHERE container_id=?", r -> r.getLong(0), voidPop(),
					container.id);
		}
		if (filter.matchUnseenVisible()) {
			return unique("SELECT unseen FROM cnt_items_unseen_visible WHERE container_id=?", r -> r.getLong(0),
					voidPop(), container.id);
		}

		throw new UnsupportedOperationException("Filter is too complex");
	}

	@Override
	public Item setExtId(String uid, String extId) throws SQLException {
		Item forUpdate = get(uid);
		forUpdate(forUpdate);
		forUpdate.externalId = extId;
		return update(forUpdate, forUpdate.displayName, forUpdate.flags);
	}

	@Override
	public Item update(String uid, String displayName, Collection<ItemFlag> flags) throws SQLException {
		return update(itemId(uid), displayName, flags);
	}

	@Override
	public Item update(Item forUpdate, String displayName, Collection<ItemFlag> flags) throws SQLException {
		Item old = get(forUpdate.uid);

		forUpdate.displayName = displayName;
		forUpdate.flags = flags == null || flags.isEmpty() ? EnumSet.noneOf(ItemFlag.class) : EnumSet.copyOf(flags);
		if (forUpdate.updated == null) {
			forUpdate.updated = new Date();
		}
		if (forUpdate.updatedBy == null) {
			forUpdate.updatedBy = ctx.getSubject();
		}
		if (forUpdate.id == 0) {
			forUpdate.id = old.id;
		}
		if (forUpdate.version == 0) {
			forUpdate.version = nextVersion();
		}
		PreparedStatement prep = session.prepare("""
				UPDATE t_container_item
				SET
				version = ?,
				updated = ?,
				updatedby = ?,
				external_id = ?,
				display_name = ?,
				flag_seen = ?,
				flag_deleted = ?,
				flag_important = ?
				WHERE container_id = ? AND id = ?
				""");

		BoundStatement bound = prep.bind((int) forUpdate.version, forUpdate.updated.toInstant(), forUpdate.updatedBy,
				forUpdate.externalId, displayName, forUpdate.flags.contains(ItemFlag.Seen),
				forUpdate.flags.contains(ItemFlag.Deleted), forUpdate.flags.contains(ItemFlag.Important), container.id,
				forUpdate.id);
		session.execute(bound);

		if ((old.visible() && old.unseen()) && !forUpdate.visible()) {
			updateUnseenVisibleCount(-1);
		} else if (old.visible() && forUpdate.visible() && !old.unseen() && forUpdate.unseen()) {
			updateUnseenVisibleCount(+1);
		}
		if (old.unseen() != forUpdate.unseen()) {
			updateUnseenCount(forUpdate.unseen() ? +1 : -1);
		}

		return forUpdate;
	}

	@Override
	public Item update(long id, String displayName, Collection<ItemFlag> flags) throws SQLException {
		Item forUpdate = getById(id);
		forUpdate(forUpdate);
		return update(forUpdate, displayName, flags);
	}

	private void forUpdate(Item forUpdate) {
		forUpdate.version = nextVersion();
		forUpdate.updated = new Date();
		forUpdate.updatedBy = ctx.getSubject();
	}

	@Override
	public void touchContainer(ItemVersion version) throws SQLException {
		seqs.setMinimumValue(version.version);
	}

	@Override
	public Item touch(String uid) throws SQLException {
		Item forUpdate = get(uid);
		forUpdate(forUpdate);
		PreparedStatement prep = session.prepare("""
				UPDATE t_container_item
				SET
				version = ?,
				updated = ?,
				updatedby = ?
				WHERE container_id = ? AND id = ?
				""");
		ResultSet result = session.execute(prep.bind((int) forUpdate.version, forUpdate.updated.toInstant(),
				forUpdate.updatedBy, container.id, forUpdate.id));
		if (!result.wasApplied()) {
			throw new CqlPersistenceException("nothing to update for uid " + forUpdate.uid);
		}
		return forUpdate;
	}

	@Override
	public Item get(String uid) {
		long itemId = itemId(uid);
		if (itemId == 0) {
			return null;
		}
		return getById(itemId);
	}

	@Override
	public Item getByExtId(String extId) {
		PreparedStatement prep = session.prepare("""
				SELECT id FROM t_container_item_by_ext_id
				WHERE container_id = ? AND external_id = ?
				""");
		List<Item> allItems = session.execute(prep.bind(container.id, extId)).map(r -> getById(r.getLong(0))).all();
		if (allItems.size() > 1) {
			throw new CqlFailedConstraintException("More than one item with external_id " + extId);
		} else if (allItems.isEmpty()) {
			return null;
		}
		return allItems.get(0);
	}

	@Override
	public Item getById(long itemId) {
		PreparedStatement prep = session.prepare("""
				SELECT
				uid, version, external_id, display_name, createdby, updatedby, created, updated,
				flag_seen, flag_deleted, flag_important
				FROM t_container_item
				WHERE
				container_id = ? AND id = ?
				""");
		ResultSet result = session.execute(prep.bind(container.id, itemId));
		Row r = result.one();
		if (r == null) {
			return null;
		}
		Item it = new Item();
		it.id = itemId;
		POP.populate(r, 0, it);
		return it;
	}

	@Override
	public List<Item> getMultiple(List<String> uids) throws SQLException {
		List<Long> fullIdsList = new ArrayList<>();
		for (List<String> slice : Lists.partition(uids, 64)) {
			List<Long> asIds = map("""
					SELECT
					id
					FROM t_container_item_by_uid
					WHERE
					container_id = ? AND uid IN ?
					""", r -> r.getLong(0), voidPop(), container.id, slice);
			fullIdsList.addAll(asIds);
		}
		return getMultipleById(fullIdsList);
	}

	@Override
	public List<Item> getMultipleById(List<Long> ids) {
		List<Item> fullItemsList = new ArrayList<>();
		for (List<Long> slice : Lists.partition(ids, 64)) {
			List<Item> tmp = map("""
					SELECT
					id, uid, version, external_id, display_name, createdby, updatedby, created, updated,
					flag_seen, flag_deleted, flag_important
					FROM t_container_item
					WHERE
					container_id = ? AND id IN ?
					""", r -> {
				Item it = new Item();
				it.id = r.getLong(0);
				return it;
			}, (r, i, v) -> POP.populate(r, i + 1, v), container.id, slice);
			fullItemsList.addAll(tmp);
		}
		return fullItemsList;
	}

	@Override
	public List<Item> all() throws SQLException {
		return map("""
				SELECT
				id, uid, version, external_id, display_name, createdby, updatedby, created, updated,
				flag_seen, flag_deleted, flag_important
				FROM t_container_item
				WHERE
				container_id = ?
				""", r -> {
			Item it = new Item();
			it.id = r.getLong(0);
			return it;
		}, (r, i, v) -> POP.populate(r, i + 1, v), container.id);
	}

	@Override
	public int getItemCount() throws SQLException {
		return (int) count(ItemFlagFilter.all());
	}

	@Override
	public void delete(Item item) throws SQLException {
		delete0(item);
	}

	private long delete0(Item item) {
		Item prev = item;
		if (prev.id == 0) {
			prev = get(prev.uid);
		}

		if (prev != null) {
			voidCql("""
					DELETE FROM t_container_item
					WHERE container_id = ? AND id = ?
					""", container.id, prev.id);
			if (prev.unseen()) {
				updateUnseenCount(-1);
				updateUnseenVisibleCount(-1);
			}
			updateTotalCount(-1);
		}
		return seqs.nextVal(Sequences.ownerIds(container.owner));
	}

	@Override
	public void deleteAll() throws SQLException {
		voidCql("DELETE FROM t_container_item WHERE container_id = ?", container.id);
	}

	@Override
	public List<String> allItemUids() throws SQLException {
		return map("""
				SELECT
				uid
				FROM t_container_item_by_uid
				WHERE
				container_id = ?
				""", r -> r.getString(0), voidPop(), container.id);

	}

	@Override
	public List<Long> allItemIds() throws SQLException {
		return map("""
				SELECT
				id
				FROM t_container_item
				WHERE
				container_id = ?
				""", r -> r.getLong(0), voidPop(), container.id);
	}

	@Override
	public long getVersion() throws SQLException {
		return seqs.curVal(Sequences.itemVersions(container.uid));
	}

	private static final EntityPopulator<Item> POP = (Row r, int i, Item it) -> {
		it.uid = r.getString(i++);
		it.version = r.getInt(i++);
		it.externalId = r.getString(i++);
		it.displayName = r.getString(i++);
		it.createdBy = r.getString(i++);
		it.updatedBy = r.getString(i++);
		it.created = Date.from(r.getInstant(i++));
		it.updated = Date.from(r.getInstant(i++));
		EnumSet<ItemFlag> fl = EnumSet.noneOf(ItemFlag.class);
		if (r.getBoolean(i++)) {
			fl.add(ItemFlag.Seen);
		}
		if (r.getBoolean(i++)) {
			fl.add(ItemFlag.Deleted);
		}
		if (r.getBoolean(i++)) {
			fl.add(ItemFlag.Important);
		}
		it.flags = fl;
		return i;
	};

}
