/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BatchStatement;
import com.datastax.oss.driver.api.core.cql.BatchStatementBuilder;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.DefaultBatchType;
import com.datastax.oss.driver.api.core.cql.Row;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IAclStore;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlAclStore extends CqlAbstractStore implements IAclStore {

	private static final EntityPopulator<List<AccessControlEntry>> ACL_POP = (Row r, int index,
			List<AccessControlEntry> value) -> {
		String subject = r.getString(index++);
		Set<String> verbs = r.getSet(index++, String.class);
		verbs.stream().forEach(v -> {
			Verb verb = Verb.valueOf(v);
			AccessControlEntry ace = new AccessControlEntry();
			ace.subject = subject;
			ace.verb = verb;
			value.add(ace);
		});
		return index;
	};

	public CqlAclStore(CqlSession s) {
		super(s);
	}

	@Override
	public void store(Container container, List<AccessControlEntry> entries) throws ServerFault {
		deleteAll(container);
		add(container, entries);
	}

	@Override
	public void add(Container container, List<AccessControlEntry> entries) {
		BatchStatementBuilder batch = BatchStatement.builder(DefaultBatchType.LOGGED);
		for (var ace : entries) {
			BoundStatement bound = session.prepare("""
					UPDATE t_container_acl
					SET verbs = verbs + ?
					WHERE container_id = ? AND subject = ?
					""").bind(Set.of(ace.verb.name()), container.id, ace.subject);
			batch.addStatement(bound);
		}
		session.execute(batch.build());
	}

	@Override
	public List<AccessControlEntry> get(Container container) {
		List<List<AccessControlEntry>> aces = map("""
				SELECT subject, verbs FROM t_container_acl
				WHERE container_id = ?
				""", r -> new ArrayList<AccessControlEntry>(), ACL_POP, container.id);
		List<AccessControlEntry> full = new ArrayList<>();
		for (var l : aces) {
			full.addAll(l);
		}
		return full;
	}

	@Override
	public void deleteAll(Container container) {
		voidCql("DELETE FROM t_container_acl WHERE container_id=?", container.id);
	}

	@Override
	public List<AccessControlEntry> retrieveAndStore(Container container, List<AccessControlEntry> entries)
			throws ServerFault {
		List<AccessControlEntry> toReturn = get(container);
		store(container, entries);
		return toReturn;
	}

}
