/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.service.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.List;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.api.GenerationContent;
import net.bluemind.dataprotect.api.GenerationIndex;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.domain.api.Domain;
import net.bluemind.utils.FileUtils;

/**
 * Used to read directory entries from "new" JSON backup format
 */
public class DirEntriesJsonWorker implements IDirEntriesWorker {
	static final Logger logger = LoggerFactory.getLogger(DirEntriesJsonWorker.class);

	private final ParGenWorkerRecord partGen;

	public DirEntriesJsonWorker(ParGenWorkerRecord partGen) {
		this.partGen = partGen;
	}

	@Override
	public String directoryDataType() {
		return WorkerDataType.DIRECTORY.value;
	}

	@Override
	public String workFilePath(String serverIp) {
		return String.format("/var/backups/bluemind/dp_spool/rsync/%s/bm/core/%d/var/backups/bluemind/directory/",
				serverIp, partGen.directory().id);
	}

	@Override
	public void loadContent(String serverIp, GenerationContent geneneration) {
		String path = workFilePath(serverIp);
		List<File> files = loadFiles(path);

		logger.debug("Read files {}", files.stream().map(File::getName).toList());

		for (File domainJson : files) {
			String domainUid = null;
			try {
				// "___<domainUid>.json"
				domainUid = domainJson.getName().substring(3).split(".json")[0];
			} catch (Exception e) {
				throw new ServerFault("Domain not found");
			}

			logger.debug("Consuming for domain {}", domainUid);

			boolean isDomainGlobal = partGen.ctx().getSecurityContext().isDomainGlobal();
			boolean isPartGenDomain = domainUid.equals(partGen.ctx().getSecurityContext().getContainerUid());
			if (!isDomainGlobal && !isPartGenDomain) {
				continue;
			}

			Domain domain = new Domain();
			domain.name = domainUid;
			geneneration.domains.add(ItemValue.create(domainUid, domain));

			try (InputStream in = new FileInputStream(domainJson)) {
				String data = FileUtils.streamString(in, true);
				List<GenerationIndex> dirEntries = JsonUtils.read(data, DirEntries.class).entries();
				geneneration.entries.addAll(dirEntries);
			} catch (Exception e) {
				throw new ServerFault("Cannot read domain json file " + domainJson.getName(), e);
			}
		}
	}

	@Override
	public List<File> loadFiles(String path) {
		File dir = new File(path);
		if (!dir.exists()) {
			throw new ServerFault("Unable to load directory listing: '" + path + "' does not exists");
		}
		return Stream.of(dir.listFiles(File::isFile)).filter(f -> f.getName().startsWith("___")).toList();
	}

	@Override
	public ParGenWorkerRecord getPartGenRecord() {
		return partGen;
	}

}
