/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lmtp.filter.imip;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.VEventOccurrence;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.delivery.lmtp.common.LmtpAddress;
import net.bluemind.delivery.lmtp.common.ResolvedBox;
import net.bluemind.delivery.lmtp.filters.PermissionDeniedException.MailboxInvitationDeniedException;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.domain.api.Domain;
import net.bluemind.icalendar.api.ICalendarElement;
import net.bluemind.icalendar.api.ICalendarElement.Classification;
import net.bluemind.imip.parser.IMIPInfos;
import net.bluemind.lmtp.filter.imip.EventCancelHandler.EventCancellationHandler.IEventCancellation;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

/**
 * Handles cancellations of meetings
 * 
 * 
 */
public class EventCancelHandler extends CancelHandler implements IIMIPHandler {

	public EventCancelHandler(ResolvedBox recipient, LmtpAddress sender) {
		super(recipient, sender);
	}

	private static final Logger logger = LoggerFactory.getLogger(EventCancelHandler.class);

	@Override
	public IMIPResponse handle(IMIPInfos imip, ResolvedBox recipient, ItemValue<Domain> domain,
			ItemValue<Mailbox> recipientMailbox) throws ServerFault {

		if (!super.validate(imip)) {
			return IMIPResponse.createEmptyResponse();
		}

		try {
			String calUid = getCalendarUid(recipientMailbox);

			// BM-2892 invitation right
			IUser userService = provider().instance(IUser.class, recipient.getDomainPart());
			ItemValue<User> sender = userService.byEmail(imip.organizerEmail);
			if (sender != null) {
				boolean canInvite = checkInvitationRight(recipient, calUid, sender);
				if (!canInvite) {
					ServerFault fault = new ServerFault(new MailboxInvitationDeniedException(recipientMailbox.uid));
					fault.setCode(ErrorCode.PERMISSION_DENIED);
					throw fault;
				}

			} // else external, don't care for now
			ICalendar cal = provider().instance(ICalendar.class, calUid);
			VEventSeries series = fromList(imip.properties, imip.iCalendarElements, imip.uid);
			List<ItemValue<VEventSeries>> currentSeries = cal.getByIcsUid(imip.uid);
			if (currentSeries.isEmpty()) {
				logger.warn("BM VEvent with event uid {} not found in calendar {}", imip.uid, calUid);
				return IMIPResponse.createEmptyResponse();
			}

			IEventCancellation handler = EventCancellationHandler.get(recipient.entry.kind, imip, cal, calUid);

			if (series.main == null) {
				List<VEventOccurrence> occurrences = series.occurrences;
				if (currentSeries.size() == 1 && currentSeries.get(0).value.main != null) {
					handler.cancelOccurrences(currentSeries, occurrences);
				} else {
					for (VEventOccurrence occurrence : occurrences) {
						for (ItemValue<VEventSeries> occurenceSeries : currentSeries) {
							VEventOccurrence oneOccurrence = occurenceSeries.value.occurrence(occurrence.recurid);
							if (occurenceSeries.value.occurrence(occurrence.recurid) != null) {
								handler.cancelOrphanOccurrence(oneOccurrence, occurrence, occurenceSeries);
							}
						}
					}
				}
			} else {
				logger.info("[{}] Cancelling BM Event with ics uid {} in calendar {} ", imip.messageId, imip.uid,
						calUid);
				for (ItemValue<VEventSeries> oneOccurence : currentSeries) {
					handler.cancelSeries(oneOccurence, series);
				}
			}

			if (imipMessageContainsASingleException(series)) {
				VEventOccurrence vEventOccurrence = series.occurrences.get(0);
				return IMIPResponse.createCanceledExceptionResponse(imip.uid, vEventOccurrence.recurid.iso8601, calUid,
						vEventOccurrence.classification == Classification.Private);
			} else {
				return IMIPResponse.createCanceledResponse(imip.uid, calUid,
						series.flatten().get(0).classification == Classification.Private);
			}
		} catch (Exception e) {
			throw e;
		}
	}

	private boolean imipMessageContainsASingleException(VEventSeries series) {
		return series.occurrences.size() == 1 && series.main == null;
	}

	public static class EventCancellationHandler {

		public static IEventCancellation get(BaseDirEntry.Kind kind, IMIPInfos infos, ICalendar cal, String calUid) {
			return switch (kind) {
			case RESOURCE: {
				yield new ResourceEventCancellationHandler(cal, infos, calUid);
			}
			default: {
				yield new UserEventCancellationHandler(cal, infos, calUid);
			}
			};
		}

		public interface IEventCancellation {

			public void cancelOccurrences(List<ItemValue<VEventSeries>> dbSeries, List<VEventOccurrence> occurrences);

			public void cancelOrphanOccurrence(VEventOccurrence dbOccurrence, VEventOccurrence occurrence,
					ItemValue<VEventSeries> occurenceSeries);

			public void cancelSeries(ItemValue<VEventSeries> oneOccurence, VEventSeries series);

		}

		public static class UserEventCancellationHandler implements IEventCancellation {

			private final IMIPInfos infos;
			private final ICalendar cal;
			private final String calUid;

			public UserEventCancellationHandler(ICalendar cal, IMIPInfos infos, String calUid) {
				this.infos = infos;
				this.cal = cal;
				this.calUid = calUid;
			}

			@Override
			public void cancelOccurrences(List<ItemValue<VEventSeries>> dbSeries, List<VEventOccurrence> occurrences) {
				ItemValue<VEventSeries> master = dbSeries.get(0);
				for (VEventOccurrence occurrence : occurrences) {
					logger.info("[{}] Cancelling event exception id {}, reccurid {}, in calendar {} ", infos.messageId,
							infos.uid, occurrence.recurid, calUid);
					VEventOccurrence occ = master.value.occurrence(occurrence.recurid);
					if (occ != null) {
						occ.status = ICalendarElement.Status.Cancelled;
						occ.sequence = occurrence.sequence;
					} else {
						if (master.value.occurrences.isEmpty()) {
							master.value.occurrences = Arrays.asList(occurrence);
						} else {
							master.value.occurrences.add(occurrence);
						}
					}
				}
				cal.update(master.uid, master.value, false);
			}

			public void cancelOrphanOccurrence(VEventOccurrence dbOccurrence, VEventOccurrence occurrence,
					ItemValue<VEventSeries> occurenceSeries) {
				dbOccurrence.status = ICalendarElement.Status.Cancelled;
				dbOccurrence.sequence = occurrence.sequence;
				cal.update(occurenceSeries.uid, occurenceSeries.value, false);
			}

			@Override
			public void cancelSeries(ItemValue<VEventSeries> oneOccurence, VEventSeries series) {
				oneOccurence.value.flatten().forEach(event -> {
					event.status = ICalendarElement.Status.Cancelled;
				});
				oneOccurence.value.main.sequence = series.main.sequence;
				cal.update(oneOccurence.uid, oneOccurence.value, false);
			}

		}

		public static class ResourceEventCancellationHandler implements IEventCancellation {

			private final IMIPInfos infos;
			private final ICalendar cal;
			private final String calUid;

			public ResourceEventCancellationHandler(ICalendar cal, IMIPInfos infos, String calUid) {
				this.infos = infos;
				this.cal = cal;
				this.calUid = calUid;
			}

			@Override
			public void cancelOccurrences(List<ItemValue<VEventSeries>> dbSeries, List<VEventOccurrence> occurrences) {
				ItemValue<VEventSeries> master = dbSeries.get(0);
				for (VEventOccurrence occurrence : occurrences) {
					logger.info("[{}] Deleting resource event exception id {}, reccurid {}, in calendar {} ",
							infos.messageId, infos.uid, occurrence.recurid, calUid);
					List<VEventOccurrence> currentOccurrences = new ArrayList<>(master.value.occurrences);
					master.value.occurrences = master.value.occurrences.stream()
							.filter(o -> !(o.recurid.equals(occurrence.recurid))).toList();
					currentOccurrences.removeAll(master.value.occurrences);
					if (master.value.main != null) {
						master.value.main.exdate = master.value.main.exdate != null
								? new HashSet<>(master.value.main.exdate)
								: new HashSet<>();
						master.value.main.exdate.add(occurrence.recurid);
					}
				}
				cal.update(master.uid, master.value, false);
			}

			@Override
			public void cancelOrphanOccurrence(VEventOccurrence dbOccurrence, VEventOccurrence occurrence,
					ItemValue<VEventSeries> occurenceSeries) {
				logger.info("[{}] Deleting orphan resource event exception id {}, reccurid {}, in calendar {} ",
						infos.messageId, infos.uid, occurrence.recurid, calUid);
				occurenceSeries.value.occurrences = occurenceSeries.value.occurrences.stream()
						.filter(o -> !(o.recurid.equals(occurrence.recurid))).toList();
				if (occurenceSeries.value.main == null && occurenceSeries.value.occurrences.isEmpty()) {
					cal.delete(occurenceSeries.uid);
				} else {
					cal.update(occurenceSeries.uid, occurenceSeries.value, false);
				}
			}

			@Override
			public void cancelSeries(ItemValue<VEventSeries> oneOccurence, VEventSeries series) {
				cal.delete(oneOccurence.uid, false);
			}

		}
	}

}
