/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.cli.auditlog;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.github.freva.asciitable.AsciiTable;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogQuery;
import net.bluemind.core.auditlogs.api.ILogRequestService;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import picocli.CommandLine.Option;

public abstract class AbstractGetAuditLogCommand implements Runnable, ICmdLet {

	enum OutputFormat {
		json, table
	}

	protected CliContext ctx;

	@Option(names = "--domain", required = false, description = "domain uid or domain aliases (default ${DEFAULT-VALUE})")
	public String domainUid = "global.virt";

	@Option(names = "--raw", required = false, description = "option to have full output data")
	public boolean raw;

	@Option(names = "--output", description = "output format to use (default ${DEFAULT-VALUE}): ${COMPLETION-CANDIDATES}")
	public OutputFormat outputFormat = OutputFormat.table;

	@Option(names = "--size", description = "number of entries to retrieve (default ${DEFAULT-VALUE})")
	public int size = 10;

	@Option(names = "--containerUid", description = "the container uid to retrieve data from")
	public String containerUid;

	@Option(names = "--owner", description = "the container owner address mail")
	public String owner;

	@Option(names = "--after", required = false, description = "get entries older than given date, format: 'yyyy-MM-dd HH:mm:ss' or 'yyyy-MM-dd'")
	public String after;

	@Option(names = "--before", required = false, description = "get entries younger than given date, format: 'yyyy-MM-dd HH:mm:ss' or 'yyyy-MM-dd'")
	public String before;

	ItemValue<Domain> domain;

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	@Override
	public void run() {
		ILogRequestService service = ctx.adminApi().instance(ILogRequestService.class);

		domain = ctx.adminApi().instance(IDomains.class).findByNameOrAliases(domainUid);
		if (domain == null) {
			ctx.error("domain '" + domainUid + "' cannot be found");
			throw new CliException("domain '" + domainUid + "' cannot be found");
		}
		AuditLogQuery auditLogQuery = buildQuery(domain);

		List<AuditLogEntry> results = service.queryAuditLog(auditLogQuery);
		if (raw) {
			results.forEach(e -> ctx.info(JsonUtils.asString(e)));
			return;
		}

		if (outputFormat == OutputFormat.table) {
			ctx.info(AsciiTable.getTable(headers(), generateOutputTable(results)));
		} else {
			results.forEach(e -> ctx.info(JsonUtils.asString(generateOutputJSON(e))));
		}

	}

	protected abstract String[][] generateOutputTable(List<AuditLogEntry> logMailQueryResults);

	protected abstract Map<String, String> generateOutputJSON(AuditLogEntry logMailQueryResults);

	protected abstract AuditLogQuery buildQuery(ItemValue<Domain> domain);

	protected abstract String[] headers();

	protected abstract String[] logTypes();

	protected AuditLogQuery defaultQuery() throws CliException {
		AuditLogQuery auditLogQuery = new AuditLogQuery();
		auditLogQuery.size = size;
		auditLogQuery.domainUid = domain.uid;
		auditLogQuery.logtypes = logTypes();
		if (containerUid != null && !containerUid.isBlank()) {
			auditLogQuery.containerUid = containerUid;
		}

		if (owner != null && !owner.isBlank()) {

			IDirectory dirService = ctx.adminApi().instance(IDirectory.class, domain.uid);
			DirEntry dirEntry = dirService.getByEmail(owner);
			if (dirEntry == null) {
				ctx.error("resource with email '" + owner + "'cannot be found");
				throw new CliException("resource with email '" + owner + "'cannot be found");
			}
			auditLogQuery.containerOwner = dirEntry.entryUid;
		}

		if (after != null) {
			auditLogQuery.before = getTimestamp(after);
		}

		if (before != null) {
			auditLogQuery.before = getTimestamp(before);
		}
		return auditLogQuery;
	}

	private long getTimestamp(String date) {
		try {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			Date startDate = formatter.parse(date);
			return startDate.getTime();
		} catch (ParseException e) {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
			try {
				Date formattedDate = formatter.parse(before);
				return formattedDate.getTime();
			} catch (ParseException e1) {
				e1.printStackTrace();
				return (new Date()).getTime();
			}
		}
	}
}
