/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.indexing.incremental.vcard;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.persistence.VCardIndexStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexerByUid;
import net.bluemind.indexing.incremental.AbstractTypeIndexerByUid.UidIndexWriter;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.lib.elasticsearch.ESearchActivator;

public class VCardsIncrementalIndexFactory implements TypeIndexerFactory {

	public String type() {
		return IAddressBookUids.TYPE;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		IAddressBook addressBook = ctx.provider().instance(IAddressBook.class, c.uid);
		ElasticsearchClient esClient = ESearchActivator.getClient();
		VCardIndexStore indexStore = new VCardIndexStore(esClient, c, DataSourceRouter.location(ctx, c.uid));
		return new VCardIndexer(addressBook, new VCardIndexWriter(indexStore));
	}

	public static class VCardIndexWriter implements UidIndexWriter<VCard> {

		private VCardIndexStore indexStore;

		public VCardIndexWriter(VCardIndexStore cidx) {
			this.indexStore = cidx;
		}

		@Override
		public void write(ItemValue<VCard> toIndex) {
			indexStore.update(toIndex.item(), toIndex.value);

		}

		@Override
		public void deleteByUid(String uid, String type) {
			indexStore.delete(uid);
		}
	}

	public static class VCardIndexer extends AbstractTypeIndexerByUid<VCard> {

		public VCardIndexer(IAddressBook rec, UidIndexWriter<VCard> writer) {
			super(rec, rec, writer);
		}

	}

}
